"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateName = exports.removeUndefined = exports.calculateVersion = exports.calculateInterval = void 0;

var _fp = require("lodash/fp");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */


const calculateInterval = (interval, ruleInterval) => {
  if (interval != null) {
    return interval;
  } else if (ruleInterval != null) {
    return ruleInterval;
  } else {
    return '5m';
  }
};

exports.calculateInterval = calculateInterval;

const calculateVersion = (immutable, currentVersion, updateProperties) => {
  // early return if we are pre-packaged/immutable rule to be safe. We are never responsible
  // for changing the version number of an immutable. Immutables are only responsible for changing
  // their own version number. This would be really bad if an immutable version number is bumped by us
  // due to a bug, hence the extra check and early bail if that is detected.
  if (immutable === true) {
    if (updateProperties.version != null) {
      // we are an immutable rule but we are asking to update the version number so go ahead
      // and update it to what is asked.
      return updateProperties.version;
    } else {
      // we are immutable and not asking to update the version number so return the existing version
      return currentVersion;
    }
  } // white list all properties but the enabled/disabled flag. We don't want to auto-increment
  // the version number if only the enabled/disabled flag is being set. Likewise if we get other
  // properties we are not expecting such as updatedAt we do not to cause a version number bump
  // on that either.


  const removedNullValues = removeUndefined(updateProperties);

  if ((0, _fp.isEmpty)(removedNullValues)) {
    return currentVersion;
  } else {
    return currentVersion + 1;
  }
};

exports.calculateVersion = calculateVersion;

const removeUndefined = obj => {
  return (0, _fp.pickBy)(value => value != null, obj);
};

exports.removeUndefined = removeUndefined;

const calculateName = ({
  updatedName,
  originalName
}) => {
  if (updatedName != null) {
    return updatedName;
  } else if (originalName != null) {
    return originalName;
  } else {
    // You really should never get to this point. This is a fail safe way to send back
    // the name of "untitled" just in case a rule name became null or undefined at
    // some point since TypeScript allows it.
    return 'untitled';
  }
};

exports.calculateName = calculateName;