"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.flattenHitWrapper = flattenHitWrapper;
exports.createFlattenHitWrapper = createFlattenHitWrapper;

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
// Takes a hit, merges it with any stored/scripted fields, and with the metaFields
// returns a flattened version
function flattenHit(indexPattern, hit, deep) {
  const flat = {}; // recursively merge _source

  const fields = indexPattern.fields.getByName;

  (function flatten(obj, keyPrefix = '') {
    keyPrefix = keyPrefix ? keyPrefix + '.' : '';

    _lodash.default.forOwn(obj, function (val, key) {
      key = keyPrefix + key;

      if (deep) {
        const field = fields(key);
        const isNestedField = field && field.type === 'nested';

        const isArrayOfObjects = Array.isArray(val) && _lodash.default.isPlainObject(_lodash.default.first(val));

        if (isArrayOfObjects && !isNestedField) {
          _lodash.default.each(val, v => flatten(v, key));

          return;
        }
      } else if (flat[key] !== void 0) {
        return;
      }

      const field = fields(key);
      const hasValidMapping = field && field.type !== 'conflict';
      const isValue = !_lodash.default.isPlainObject(val);

      if (hasValidMapping || isValue) {
        if (!flat[key]) {
          flat[key] = val;
        } else if (Array.isArray(flat[key])) {
          flat[key].push(val);
        } else {
          flat[key] = [flat[key], val];
        }

        return;
      }

      flatten(val, key);
    });
  })(hit._source);

  return flat;
}

function decorateFlattenedWrapper(hit, metaFields) {
  return function (flattened) {
    // assign the meta fields
    _lodash.default.each(metaFields, function (meta) {
      if (meta === '_source') return;
      flattened[meta] = hit[meta];
    }); // unwrap computed fields


    _lodash.default.forOwn(hit.fields, function (val, key) {
      if (key[0] === '_' && !_lodash.default.includes(metaFields, key)) return; // Flatten an array with 0 or 1 elements to a single value.

      if (Array.isArray(val) && val.length <= 1) {
        flattened[key] = val[0];
      } else {
        flattened[key] = val;
      }
    }); // Force all usage of Object.keys to use a predefined sort order,
    // instead of using insertion order


    return new Proxy(flattened, {
      ownKeys: target => {
        return Reflect.ownKeys(target).sort((a, b) => {
          const aIsMeta = _lodash.default.includes(metaFields, a);

          const bIsMeta = _lodash.default.includes(metaFields, b);

          if (aIsMeta && bIsMeta) {
            return String(a).localeCompare(String(b));
          }

          if (aIsMeta) {
            return 1;
          }

          if (bIsMeta) {
            return -1;
          }

          return String(a).localeCompare(String(b));
        });
      }
    });
  };
}
/**
 * This is wrapped by `createFlattenHitWrapper` in order to provide a single cache to be
 * shared across all uses of this function. It is only exported here for use in mocks.
 *
 * @internal
 */


function flattenHitWrapper(indexPattern, metaFields = {}, cache = new WeakMap()) {
  return function cachedFlatten(hit, deep = false) {
    const decorateFlattened = decorateFlattenedWrapper(hit, metaFields);
    const cached = cache.get(hit);
    const flattened = cached || flattenHit(indexPattern, hit, deep);

    if (!cached) {
      cache.set(hit, { ...flattened
      });
    }

    return decorateFlattened(flattened);
  };
}
/**
 * This wraps `flattenHitWrapper` so one single cache can be provided for all uses of that
 * function. The returned value of this function is what is included in the index patterns
 * setup contract.
 *
 * @public
 */


function createFlattenHitWrapper() {
  const cache = new WeakMap();
  return _lodash.default.partial(flattenHitWrapper, _lodash.default, _lodash.default, cache);
}