"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.registerGrammarChecker = void 0;
// This file contains a lot of single setup logic for registering a language globally
const monaco_imports_1 = require("../monaco_imports");
const worker_proxy_service_1 = require("./worker_proxy_service");
const constants_1 = require("./constants");
const wps = new worker_proxy_service_1.WorkerProxyService();
monaco_imports_1.monaco.languages.onLanguage(constants_1.ID, async () => {
    return wps.setup();
});
const OWNER = 'XJSON_GRAMMAR_CHECKER';
const registerGrammarChecker = () => {
    const allDisposables = [];
    const updateAnnotations = async (model) => {
        if (model.isDisposed()) {
            return;
        }
        const parseResult = await wps.getAnnos(model.uri);
        if (!parseResult) {
            return;
        }
        const { annotations } = parseResult;
        monaco_imports_1.monaco.editor.setModelMarkers(model, OWNER, annotations.map(({ at, text, type }) => {
            const { column, lineNumber } = model.getPositionAt(at);
            return {
                startLineNumber: lineNumber,
                startColumn: column,
                endLineNumber: lineNumber,
                endColumn: column,
                message: text,
                severity: type === 'error' ? monaco_imports_1.monaco.MarkerSeverity.Error : monaco_imports_1.monaco.MarkerSeverity.Warning,
            };
        }));
    };
    const onModelAdd = (model) => {
        if (model.getModeId() === constants_1.ID) {
            allDisposables.push(model.onDidChangeContent(async () => {
                updateAnnotations(model);
            }));
            updateAnnotations(model);
        }
    };
    allDisposables.push(monaco_imports_1.monaco.editor.onDidCreateModel(onModelAdd));
    return () => {
        wps.stop();
        allDisposables.forEach((d) => d.dispose());
    };
};
exports.registerGrammarChecker = registerGrammarChecker;
exports.registerGrammarChecker();
//# sourceMappingURL=language.js.map