"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAutocompleteSuggestions = exports.getConstructorSuggestions = exports.getFieldSuggestions = exports.getClassMemberSuggestions = exports.getStaticSuggestions = exports.getTypeSuggestions = exports.getKeywords = void 0;
const i18n_1 = require("@kbn/i18n");
const autocomplete_definitions_1 = require("../../autocomplete_definitions");
const lexer_rules_1 = require("../../lexer_rules");
const autocomplete_utils_1 = require("./autocomplete_utils");
const getKeywords = () => {
    const lexerKeywords = lexer_rules_1.lexerRules.keywords.map((keyword) => {
        return {
            label: keyword,
            kind: 'keyword',
            documentation: `Keyword: ${keyword}`,
            insertText: keyword,
        };
    });
    const allKeywords = [
        ...lexerKeywords,
        {
            label: 'params',
            kind: 'keyword',
            documentation: i18n_1.i18n.translate('monaco.painlessLanguage.autocomplete.paramsKeywordDescription', {
                defaultMessage: 'Access variables passed into the script.',
            }),
            insertText: 'params',
        },
    ];
    return allKeywords;
};
exports.getKeywords = getKeywords;
const getTypeSuggestions = () => {
    return lexer_rules_1.lexerRules.primitives.map((primitive) => {
        return {
            label: primitive,
            kind: 'type',
            documentation: `Type: ${primitive}`,
            insertText: primitive,
        };
    });
};
exports.getTypeSuggestions = getTypeSuggestions;
const runtimeContexts = [
    'boolean_script_field_script_field',
    'date_script_field',
    'double_script_field_script_field',
    'ip_script_field_script_field',
    'long_script_field_script_field',
    'string_script_field_script_field',
];
const mapContextToData = {
    painless_test: autocomplete_definitions_1.commonContext,
    score: autocomplete_definitions_1.scoreContext,
    filter: autocomplete_definitions_1.filterContext,
    boolean_script_field_script_field: autocomplete_definitions_1.booleanScriptFieldScriptFieldContext,
    date_script_field: autocomplete_definitions_1.dateScriptFieldContext,
    double_script_field_script_field: autocomplete_definitions_1.doubleScriptFieldScriptFieldContext,
    ip_script_field_script_field: autocomplete_definitions_1.ipScriptFieldScriptFieldContext,
    long_script_field_script_field: autocomplete_definitions_1.longScriptFieldScriptFieldContext,
    processor_conditional: autocomplete_definitions_1.processorConditionalContext,
    string_script_field_script_field: autocomplete_definitions_1.stringScriptFieldScriptFieldContext,
};
const getStaticSuggestions = ({ suggestions, hasFields, isRuntimeContext, }) => {
    const classSuggestions = suggestions.map((suggestion) => {
        const { properties, constructorDefinition, ...rootSuggestion } = suggestion;
        return rootSuggestion;
    });
    const keywords = exports.getKeywords();
    const typeSuggestions = exports.getTypeSuggestions();
    let keywordSuggestions = hasFields
        ? [
            ...keywords,
            {
                label: 'doc',
                kind: 'keyword',
                documentation: i18n_1.i18n.translate('monaco.painlessLanguage.autocomplete.docKeywordDescription', {
                    defaultMessage: `Access a field value from a script using the doc['field_name'] syntax`,
                }),
                insertText: "doc[${1:'my_field'}]",
                insertTextAsSnippet: true,
            },
        ]
        : keywords;
    keywordSuggestions = isRuntimeContext
        ? [
            ...keywordSuggestions,
            {
                label: 'emit',
                kind: 'keyword',
                documentation: i18n_1.i18n.translate('monaco.painlessLanguage.autocomplete.emitKeywordDescription', {
                    defaultMessage: 'Emit value without returning.',
                }),
                insertText: 'emit',
            },
        ]
        : keywordSuggestions;
    return {
        isIncomplete: false,
        suggestions: [...classSuggestions, ...keywordSuggestions, ...typeSuggestions],
    };
};
exports.getStaticSuggestions = getStaticSuggestions;
const getClassMemberSuggestions = (suggestions, className) => {
    const painlessClass = suggestions.find((suggestion) => suggestion.label === className);
    return {
        isIncomplete: false,
        suggestions: painlessClass?.properties || [],
    };
};
exports.getClassMemberSuggestions = getClassMemberSuggestions;
const getFieldSuggestions = (fields) => {
    const suggestions = fields.map(({ name }) => {
        return {
            label: name,
            kind: 'field',
            documentation: i18n_1.i18n.translate('monaco.painlessLanguage.autocomplete.fieldValueDescription', {
                defaultMessage: `Retrieve the value for field '{fieldName}'`,
                values: {
                    fieldName: name,
                },
            }),
            // A trailing quotation mark is added to format the field for the user
            insertText: `${name}'`,
        };
    });
    return {
        isIncomplete: false,
        suggestions,
    };
};
exports.getFieldSuggestions = getFieldSuggestions;
const getConstructorSuggestions = (suggestions) => {
    let constructorSuggestions = [];
    const suggestionsWithConstructors = suggestions.filter((suggestion) => suggestion.constructorDefinition);
    if (suggestionsWithConstructors) {
        constructorSuggestions = suggestionsWithConstructors.map((filteredSuggestion) => filteredSuggestion.constructorDefinition);
    }
    return {
        isIncomplete: false,
        suggestions: constructorSuggestions,
    };
};
exports.getConstructorSuggestions = getConstructorSuggestions;
const getAutocompleteSuggestions = (painlessContext, words, fields) => {
    // Unique suggestions based on context
    const contextSuggestions = mapContextToData[painlessContext].suggestions;
    // Enhance suggestions with common classes that exist in all contexts
    // "painless_test" is the exception since it equals the common suggestions
    const suggestions = painlessContext === 'painless_test'
        ? contextSuggestions
        : contextSuggestions.concat(autocomplete_definitions_1.commonContext.suggestions);
    // What the user is currently typing
    const activeTyping = words[words.length - 1];
    // This logic may end up needing to be more robust as we integrate autocomplete into more editors
    // For now, we're assuming there is a list of painless contexts that are only applicable in runtime fields
    const isRuntimeContext = runtimeContexts.includes(painlessContext);
    // "text" field types are not available in doc values and should be removed for autocompletion
    const filteredFields = fields?.filter((field) => field.type !== 'text');
    const hasFields = Boolean(filteredFields?.length);
    let autocompleteSuggestions = {
        isIncomplete: false,
        suggestions: [],
    };
    if (autocomplete_utils_1.isConstructorInstance(words)) {
        autocompleteSuggestions = exports.getConstructorSuggestions(suggestions);
    }
    else if (filteredFields && autocomplete_utils_1.isDeclaringField(activeTyping)) {
        autocompleteSuggestions = exports.getFieldSuggestions(filteredFields);
    }
    else if (autocomplete_utils_1.isAccessingProperty(activeTyping)) {
        const className = activeTyping.substring(0, activeTyping.length - 1).split('.')[0];
        autocompleteSuggestions = exports.getClassMemberSuggestions(suggestions, className);
    }
    else if (autocomplete_utils_1.showStaticSuggestions(activeTyping, words, lexer_rules_1.lexerRules.primitives)) {
        autocompleteSuggestions = exports.getStaticSuggestions({ suggestions, hasFields, isRuntimeContext });
    }
    return autocompleteSuggestions;
};
exports.getAutocompleteSuggestions = getAutocompleteSuggestions;
//# sourceMappingURL=autocomplete.js.map