"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DiagnosticsAdapter = void 0;
const monaco_imports_1 = require("../monaco_imports");
const constants_1 = require("./constants");
const toDiagnostics = (error) => {
    return {
        ...error,
        severity: monaco_imports_1.monaco.MarkerSeverity.Error,
    };
};
class DiagnosticsAdapter {
    constructor(worker) {
        this.worker = worker;
        const onModelAdd = (model) => {
            let handle;
            if (model.getModeId() === constants_1.ID) {
                model.onDidChangeContent(() => {
                    // Do not validate if the language ID has changed
                    if (model.getModeId() !== constants_1.ID) {
                        return;
                    }
                    // Every time a new change is made, wait 500ms before validating
                    clearTimeout(handle);
                    handle = setTimeout(() => this.validate(model.uri), 500);
                });
                model.onDidChangeLanguage(({ newLanguage }) => {
                    // Reset the model markers if the language ID has changed and is no longer "painless"
                    if (newLanguage !== constants_1.ID) {
                        return monaco_imports_1.monaco.editor.setModelMarkers(model, constants_1.ID, []);
                    }
                });
                this.validate(model.uri);
            }
        };
        monaco_imports_1.monaco.editor.onDidCreateModel(onModelAdd);
        monaco_imports_1.monaco.editor.getModels().forEach(onModelAdd);
    }
    async validate(resource) {
        const worker = await this.worker(resource);
        const errorMarkers = await worker.getSyntaxErrors(resource.toString());
        if (errorMarkers) {
            const model = monaco_imports_1.monaco.editor.getModel(resource);
            // Set the error markers and underline them with "Error" severity
            monaco_imports_1.monaco.editor.setModelMarkers(model, constants_1.ID, errorMarkers.map(toDiagnostics));
        }
    }
}
exports.DiagnosticsAdapter = DiagnosticsAdapter;
//# sourceMappingURL=diagnostics_adapter.js.map