"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.PainlessCompletionAdapter = void 0;
const monaco_imports_1 = require("../monaco_imports");
const getCompletionKind = (kind) => {
    const monacoItemKind = monaco_imports_1.monaco.languages.CompletionItemKind;
    switch (kind) {
        case 'type':
            return monacoItemKind.Interface;
        case 'class':
            return monacoItemKind.Class;
        case 'method':
            return monacoItemKind.Method;
        case 'constructor':
            return monacoItemKind.Constructor;
        case 'property':
            return monacoItemKind.Property;
        case 'keyword':
            return monacoItemKind.Keyword;
        case 'field':
            return monacoItemKind.Field;
        default:
            return monacoItemKind.Text;
    }
};
class PainlessCompletionAdapter {
    constructor(worker, editorStateService) {
        this.worker = worker;
        this.editorStateService = editorStateService;
    }
    get triggerCharacters() {
        return ['.', `'`];
    }
    async provideCompletionItems(model, position) {
        // Active line characters
        const currentLineChars = model.getValueInRange({
            startLineNumber: position.lineNumber,
            startColumn: 0,
            endLineNumber: position.lineNumber,
            endColumn: position.column,
        });
        const worker = await this.worker(model.uri);
        const { context, fields } = this.editorStateService.getState();
        const autocompleteInfo = await worker.provideAutocompleteSuggestions(currentLineChars, context, fields);
        const wordInfo = model.getWordUntilPosition(position);
        const wordRange = {
            startLineNumber: position.lineNumber,
            endLineNumber: position.lineNumber,
            startColumn: wordInfo.startColumn,
            endColumn: wordInfo.endColumn,
        };
        const suggestions = autocompleteInfo.suggestions.map(({ label, insertText, documentation, kind, insertTextAsSnippet }) => {
            return {
                label,
                insertText,
                documentation,
                range: wordRange,
                kind: getCompletionKind(kind),
                insertTextRules: insertTextAsSnippet
                    ? monaco_imports_1.monaco.languages.CompletionItemInsertTextRule.InsertAsSnippet
                    : undefined,
            };
        });
        return {
            incomplete: autocompleteInfo.isIncomplete,
            suggestions,
        };
    }
}
exports.PainlessCompletionAdapter = PainlessCompletionAdapter;
//# sourceMappingURL=completion_adapter.js.map