# Painless

This folder contains the language definitions for Painless used by the Monaco editor.

## Summary of contents

### ./language

Initializes the worker proxy service when the Painless language is first needed. It also exports the [suggestion provider](https://microsoft.github.io/monaco-editor/api/interfaces/monaco.languages.completionitemprovider.html) needed for autocompletion.

### ./lib
This directory exports two services:

1. Worker proxy: Responsible for holding a reference to the Monaco-provided proxy getter.
2. Editor state: Responsible for holding a reference to the current Painless context and fields provided as props via the React `<CodeEditor/>` component.

### ./completion_adapter

Implements the monaco [CompletionItemProvider](https://microsoft.github.io/monaco-editor/api/interfaces/monaco.languages.completionitemprovider.html). Returns autocomplete suggestions based on the editor context and current characters.

### ./constants

Contains the unique language ID.

### ./types

Exports common Typescript interfaces.

### ./lexer_rules

Contains the Monarch-specific language tokenization rules for Painless.

### ./worker

The worker proxy and worker instantiation code used in both the main thread and the worker thread. The logic for providing autocomplete suggestions and error reporting resides here.

### ./autocomplete_definitions

This directory is generated by a script and should not be changed manually. Read [Updating autocomplete definitions](#updating-autocomplete-definitions) for more information.

### ./antlr
This directory contains the Painless lexer and grammar rules, as well as the generated Typescript code. Read [Compiling ANTLR](#compiling-ANTLR) for more information.

## Example usage

```
import { PainlessLang, PainlessContext } from '@kbn/monaco';
import { CodeEditor } from '../src/plugins/kibana_react/public';

// Fields are optional and only applicable in certain contexts
const fields = [
  {
    name: 'field1',
    type: 'float',
  },
  {
    name: 'field2',
    type: 'boolean',
  },
  {
    name: 'field3',
    type: 'boolean',
  },
];

<CodeEditor
  languageId={PainlessLang.ID}
  width="100%"
  height="100%"
  value={code}
  onChange={onChange}
  suggestionProvider={PainlessLang.getSuggestionProvider(context, fields)}
  options={{
    ...,
    suggest: {
      // This is necessary to allow autocomplete suggestions for document fields
      snippetsPreventQuickSuggestions: false,
    }
  }}
/>
```

## Updating autocomplete definitions

The autocomplete definitions are generated by the `kbn-monaco/scripts/generate_autocomplete.js` script. This script should be run each release to ensure the definitions are up-to-date. 

Run from the package root directory:

```
node scripts/generate_autocomplete --tag <tag_name>
# or
node scripts/generate_autocomplete --branch <branch_name>
```

**Note:** The script currently only generates autocomplete definitions for the following contexts:

  - `boolean_script_field_script_field`
  - `date_script_field`
  - `double_script_field_script_field`
  - `filter`
  - `ip_script_field_script_field`
  - `long_script_field_script_field`
  - `painless_test`
  - `processor_conditional`
  - `score`
  - `string_script_field_script_field`

To add additional contexts, edit the `supportedContexts` constant in `kbn-monaco/scripts/constants.js`.

## Compiling ANTLR

[ANTLR](https://www.antlr.org/) generates lexical and syntax errors out of the box, which we can use to set error markers in monaco. 

Elasticsearch has defined [lexer and parser grammar](https://github.com/elastic/elasticsearch/tree/master/modules/lang-painless/src/main/antlr) for the Painless language. For now, these rules have been largely copied from ES to Kibana and reside in the `antlr` directory with the `.g4` file extension. We then use [antlr4ts](https://github.com/tunnelvisionlabs/antlr4ts) to generate a lexer and a parser in Typescript.

To regenerate the lexer and parser, run the following script:

```
npm run build:antlr4ts
```

*Note:* This script should only need to be run if a change has been made to `painless_lexer.g4` or `painless_parser.g4`.

*Note:* There is a manual change made to the `sempred()` method in the generated `painless_lexer.ts`. This needs further investigation, but it appears there is an offset between the rule index and the token value. Without this manual change, ANTLR incorrectly reports an error when using a `/` or regex in a script. There is a comment in the generated code to this effect.
