"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogInterceptor = void 0;
const tslib_1 = require("tslib");
const stream_1 = tslib_1.__importDefault(require("stream"));
const lodash_1 = require("lodash");
/**
 * Matches error messages when clients connect via HTTP instead of HTTPS; see unit test for full message. Warning: this can change when Node
 * and its bundled OpenSSL binary are upgraded.
 */
const OPENSSL_GET_RECORD_REGEX = /ssl3_get_record:http/;
function doTagsMatch(event, tags) {
    return lodash_1.isEqual(event.tags, tags);
}
function doesMessageMatch(errorMessage, match) {
    if (!errorMessage) {
        return false;
    }
    if (match instanceof RegExp) {
        return match.test(errorMessage);
    }
    return errorMessage === match;
}
// converts the given event into a debug log if it's an error of the given type
function downgradeIfErrorType(errorType, event) {
    const isClientError = doTagsMatch(event, ['connection', 'client', 'error']);
    if (!isClientError) {
        return null;
    }
    const matchesErrorType = lodash_1.get(event, 'error.code') === errorType || lodash_1.get(event, 'error.errno') === errorType;
    if (!matchesErrorType) {
        return null;
    }
    const errorTypeTag = errorType.toLowerCase();
    return {
        event: 'log',
        pid: event.pid,
        timestamp: event.timestamp,
        tags: ['debug', 'connection', errorTypeTag],
        data: `${errorType}: Socket was closed by the client (probably the browser) before it could be read completely`,
    };
}
function downgradeIfErrorMessage(match, event) {
    const isClientError = doTagsMatch(event, ['connection', 'client', 'error']);
    const errorMessage = lodash_1.get(event, 'error.message');
    const matchesErrorMessage = isClientError && doesMessageMatch(errorMessage, match);
    if (!matchesErrorMessage) {
        return null;
    }
    return {
        event: 'log',
        pid: event.pid,
        timestamp: event.timestamp,
        tags: ['debug', 'connection'],
        data: errorMessage,
    };
}
class LogInterceptor extends stream_1.default.Transform {
    constructor() {
        super({
            readableObjectMode: true,
            writableObjectMode: true,
        });
    }
    /**
     *  Since the upgrade to hapi 14, any socket read
     *  error is surfaced as a generic "client error"
     *  but "ECONNRESET" specifically is not useful for the
     *  logs unless you are trying to debug edge-case behaviors.
     *
     *  For that reason, we downgrade this from error to debug level
     *
     *  @param {object} - log event
     */
    downgradeIfEconnreset(event) {
        return downgradeIfErrorType('ECONNRESET', event);
    }
    /**
     *  Since the upgrade to hapi 14, any socket write
     *  error is surfaced as a generic "client error"
     *  but "EPIPE" specifically is not useful for the
     *  logs unless you are trying to debug edge-case behaviors.
     *
     *  For that reason, we downgrade this from error to debug level
     *
     *  @param {object} - log event
     */
    downgradeIfEpipe(event) {
        return downgradeIfErrorType('EPIPE', event);
    }
    /**
     *  Since the upgrade to hapi 14, any socket write
     *  error is surfaced as a generic "client error"
     *  but "ECANCELED" specifically is not useful for the
     *  logs unless you are trying to debug edge-case behaviors.
     *
     *  For that reason, we downgrade this from error to debug level
     *
     *  @param {object} - log event
     */
    downgradeIfEcanceled(event) {
        return downgradeIfErrorType('ECANCELED', event);
    }
    downgradeIfHTTPSWhenHTTP(event) {
        return downgradeIfErrorType('HPE_INVALID_METHOD', event);
    }
    downgradeIfHTTPWhenHTTPS(event) {
        return downgradeIfErrorMessage(OPENSSL_GET_RECORD_REGEX, event);
    }
    _transform(event, enc, next) {
        const downgraded = this.downgradeIfEconnreset(event) ||
            this.downgradeIfEpipe(event) ||
            this.downgradeIfEcanceled(event) ||
            this.downgradeIfHTTPSWhenHTTP(event) ||
            this.downgradeIfHTTPWhenHTTPS(event);
        this.push(downgraded || event);
        next();
    }
}
exports.LogInterceptor = LogInterceptor;
