"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.KbnLoggerStringFormat = void 0;
const tslib_1 = require("tslib");
const lodash_1 = tslib_1.__importDefault(require("lodash"));
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const log_format_1 = require("./log_format");
const statuses = ['err', 'info', 'error', 'warning', 'fatal', 'status', 'debug'];
const typeColors = {
    log: 'white',
    req: 'green',
    res: 'green',
    ops: 'cyan',
    config: 'cyan',
    err: 'red',
    info: 'green',
    error: 'red',
    warning: 'red',
    fatal: 'magentaBright',
    status: 'yellowBright',
    debug: 'gray',
    server: 'gray',
    optmzr: 'white',
    manager: 'green',
    optimize: 'magentaBright',
    listening: 'magentaBright',
    scss: 'magentaBright',
};
const color = lodash_1.default.memoize((name) => {
    // @ts-expect-error couldn't even get rid of the error with an any cast
    return chalk_1.default[typeColors[name]] || lodash_1.default.identity;
});
const type = lodash_1.default.memoize((t) => {
    return color(t)(lodash_1.default.pad(t, 7).slice(0, 7));
});
const prefix = process.env.isDevCliChild ? `${type('server')} ` : '';
class KbnLoggerStringFormat extends log_format_1.BaseLogFormat {
    format(data) {
        const time = color('time')(this.extractAndFormatTimestamp(data, 'HH:mm:ss.SSS'));
        const msg = data.error ? color('error')(data.error.stack) : color('message')(data.message);
        const tags = lodash_1.default(data.tags)
            .sortBy(function (tag) {
            if (color(tag) === lodash_1.default.identity)
                return `2${tag}`;
            if (lodash_1.default.includes(statuses, tag))
                return `0${tag}`;
            return `1${tag}`;
        })
            .reduce(function (s, t) {
            return s + `[${color(t)(t)}]`;
        }, '');
        return `${prefix}${type(data.type)} [${time}] ${tags} ${msg}`;
    }
}
exports.KbnLoggerStringFormat = KbnLoggerStringFormat;
