"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseLogFormat = void 0;
const tslib_1 = require("tslib");
const stream_1 = tslib_1.__importDefault(require("stream"));
const moment_timezone_1 = tslib_1.__importDefault(require("moment-timezone"));
const lodash_1 = tslib_1.__importDefault(require("lodash"));
const query_string_1 = tslib_1.__importDefault(require("query-string"));
const numeral_1 = tslib_1.__importDefault(require("@elastic/numeral"));
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const util_1 = require("util");
const utils_1 = require("./utils");
const metadata_1 = require("./metadata");
const log_events_1 = require("./log_events");
function serializeError(err = {}) {
    return {
        message: err.message,
        name: err.name,
        stack: err.stack,
        code: err.code,
        signal: err.signal,
    };
}
const levelColor = function (code) {
    if (code < 299)
        return chalk_1.default.green(String(code));
    if (code < 399)
        return chalk_1.default.yellow(String(code));
    if (code < 499)
        return chalk_1.default.magentaBright(String(code));
    return chalk_1.default.red(String(code));
};
class BaseLogFormat extends stream_1.default.Transform {
    constructor(config) {
        super({
            readableObjectMode: false,
            writableObjectMode: true,
        });
        this.config = config;
    }
    filter(data) {
        if (!this.config.filter) {
            return data;
        }
        return utils_1.applyFiltersToKeys(data, this.config.filter);
    }
    _transform(event, enc, next) {
        const data = this.filter(this.readEvent(event));
        this.push(this.format(data) + '\n');
        next();
    }
    getContentLength({ responsePayload, responseHeaders }) {
        try {
            return utils_1.getResponsePayloadBytes(responsePayload, responseHeaders);
        }
        catch (e) {
            // We intentionally swallow any errors as this information is
            // only a nicety for logging purposes, and should not cause the
            // server to crash if it cannot be determined.
            this.push(this.format({
                type: 'log',
                tags: ['warning', 'logging'],
                message: `Failed to calculate response payload bytes. [${e}]`,
            }) + '\n');
        }
    }
    extractAndFormatTimestamp(data, format) {
        const { timezone } = this.config;
        const date = moment_timezone_1.default(data['@timestamp']);
        if (timezone) {
            date.tz(timezone);
        }
        return date.format(format);
    }
    readEvent(event) {
        const data = {
            type: event.event,
            '@timestamp': event.timestamp,
            tags: [...(event.tags || [])],
            pid: event.pid,
        };
        if (log_events_1.isResponseEvent(event)) {
            lodash_1.default.defaults(data, lodash_1.default.pick(event, ['method', 'statusCode']));
            const source = lodash_1.default.get(event, 'source', {});
            data.req = {
                url: event.path,
                method: event.method || '',
                headers: event.headers,
                remoteAddress: source.remoteAddress,
                userAgent: source.userAgent,
                referer: source.referer,
            };
            data.res = {
                statusCode: event.statusCode,
                responseTime: event.responseTime,
                contentLength: this.getContentLength(event),
            };
            const query = query_string_1.default.stringify(event.query, { sort: false });
            if (query) {
                data.req.url += '?' + query;
            }
            data.message = data.req.method.toUpperCase() + ' ';
            data.message += data.req.url;
            data.message += ' ';
            data.message += levelColor(data.res.statusCode);
            data.message += ' ';
            data.message += chalk_1.default.gray(data.res.responseTime + 'ms');
            if (data.res.contentLength) {
                data.message += chalk_1.default.gray(' - ' + numeral_1.default(data.res.contentLength).format('0.0b'));
            }
        }
        else if (log_events_1.isOpsEvent(event)) {
            lodash_1.default.defaults(data, lodash_1.default.pick(event, ['pid', 'os', 'proc', 'load']));
            data.message = chalk_1.default.gray('memory: ');
            data.message += numeral_1.default(lodash_1.default.get(data, 'proc.mem.heapUsed')).format('0.0b');
            data.message += ' ';
            data.message += chalk_1.default.gray('uptime: ');
            data.message += numeral_1.default(lodash_1.default.get(data, 'proc.uptime')).format('00:00:00');
            data.message += ' ';
            data.message += chalk_1.default.gray('load: [');
            data.message += lodash_1.default.get(data, 'os.load', [])
                .map((val) => {
                return numeral_1.default(val).format('0.00');
            })
                .join(' ');
            data.message += chalk_1.default.gray(']');
            data.message += ' ';
            data.message += chalk_1.default.gray('delay: ');
            data.message += numeral_1.default(lodash_1.default.get(data, 'proc.delay')).format('0.000');
        }
        else if (log_events_1.isErrorEvent(event)) {
            data.level = 'error';
            data.error = serializeError(event.error);
            data.url = event.url;
            const message = lodash_1.default.get(event, 'error.message');
            data.message = message || 'Unknown error (no message)';
        }
        else if (log_events_1.isUndeclaredErrorEvent(event)) {
            data.type = 'error';
            data.level = lodash_1.default.includes(event.tags, 'fatal') ? 'fatal' : 'error';
            data.error = serializeError(event.error);
            const message = lodash_1.default.get(event, 'error.message');
            data.message = message || 'Unknown error object (no message)';
        }
        else if (log_events_1.isLogEvent(event)) {
            lodash_1.default.assign(data, metadata_1.getLogEventData(event.data));
        }
        else {
            data.message = lodash_1.default.isString(event.data) ? event.data : util_1.inspect(event.data);
        }
        return data;
    }
}
exports.BaseLogFormat = BaseLogFormat;
