"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegacyLoggingServer = void 0;
const tslib_1 = require("tslib");
const podium_1 = tslib_1.__importDefault(require("@hapi/podium"));
const setup_logging_1 = require("./setup_logging");
const metadata_1 = require("./metadata");
const schema_1 = require("./schema");
const isEmptyObject = (obj) => Object.keys(obj).length === 0;
function getDataToLog(error, metadata, message) {
    if (error) {
        return error;
    }
    if (!isEmptyObject(metadata)) {
        return metadata_1.attachMetaData(message, metadata);
    }
    return message;
}
/**
 * Converts core log level to a one that's known to the legacy platform.
 * @param level Log level from the core.
 */
function getLegacyLogLevel(level) {
    const logLevel = level.id.toLowerCase();
    if (logLevel === 'warn') {
        return 'warning';
    }
    if (logLevel === 'trace') {
        return 'debug';
    }
    return logLevel;
}
/**
 *  The "legacy" Kibana uses Hapi server + even-better plugin to log, so we should
 *  use the same approach here to make log records generated by the core to look the
 *  same as the rest of the records generated by the "legacy" Kibana. But to reduce
 *  overhead of having full blown Hapi server instance we create our own "light" version.
 *  @internal
 */
class LegacyLoggingServer {
    constructor(legacyLoggingConfig) {
        this.connections = [];
        // Emulates Hapi's usage of the podium event bus.
        this.events = new podium_1.default(['log', 'request', 'response']);
        // We set `ops.interval` to max allowed number and `ops` filter to value
        // that doesn't exist to avoid logging of ops at all, if turned on it will be
        // logged by the "legacy" Kibana.
        const { value: loggingConfig } = schema_1.legacyLoggingConfigSchema.validate({
            ...legacyLoggingConfig,
            events: {
                ...legacyLoggingConfig.events,
                ops: '__no-ops__',
            },
        });
        setup_logging_1.setupLogging(this, loggingConfig, 2147483647);
    }
    register({ plugin: { register }, options }) {
        return register(this, options);
    }
    log({ level, context, message, error, timestamp, meta = {} }) {
        const { tags = [], ...metadata } = meta;
        this.events
            .emit('log', {
            data: getDataToLog(error, metadata, message),
            tags: [getLegacyLogLevel(level), ...context.split('.'), ...tags],
            timestamp: timestamp.getTime(),
        })
            // @ts-expect-error @hapi/podium emit is actually an async function
            .catch((err) => {
            // eslint-disable-next-line no-console
            console.error('An unexpected error occurred while writing to the log:', err.stack);
            process.exit(1);
        });
    }
    stop() {
        // Tell the plugin we're stopping.
        if (this.onPostStopCallback !== undefined) {
            this.onPostStopCallback();
        }
    }
    ext(eventName, callback) {
        // method is called by plugin that's being registered.
        if (eventName === 'onPostStop') {
            this.onPostStopCallback = callback;
        }
        // We don't care about any others the plugin registers
    }
    expose() {
        // method is called by plugin that's being registered.
    }
}
exports.LegacyLoggingServer = LegacyLoggingServer;
