function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import * as PropTypes from 'prop-types';
import * as React from 'react';
import * as i18n from '../core';
import { isPseudoLocale, translateUsingPseudoLocale } from '../core/pseudo_locale';
/**
 * To translate label that includes nested `FormattedMessage` instances React Intl
 * replaces them with special placeholders (@__uid__@ELEMENT-uid-counter@__uid__@)
 * and maps them back with nested translations after `formatMessage` processes
 * original string, so we shouldn't modify these special placeholders with pseudo
 * translations otherwise React Intl won't be able to properly replace placeholders.
 * It's implementation detail of the React Intl, but since pseudo localization is dev
 * only feature we should be fine here.
 * @param message
 */

function translateFormattedMessageUsingPseudoLocale(message) {
  const formattedMessageDelimiter = message.match(/@__.{10}__@/);

  if (formattedMessageDelimiter !== null) {
    return message.split(formattedMessageDelimiter[0]).map(part => part.startsWith('ELEMENT-') ? part : translateUsingPseudoLocale(part)).join(formattedMessageDelimiter[0]);
  }

  return translateUsingPseudoLocale(message);
}
/**
 * If the locale is our pseudo locale (e.g. en-xa), we override the
 * intl.formatMessage function to display scrambled characters. We are
 * overriding the context rather than using injectI18n, because the
 * latter creates a new React component, which causes React diffs to
 * be inefficient in some cases, and can cause React hooks to lose
 * their state.
 */


export class PseudoLocaleWrapper extends React.PureComponent {
  constructor(props, context) {
    super(props, context);

    if (isPseudoLocale(i18n.getLocale())) {
      const formatMessage = context.intl.formatMessage;

      context.intl.formatMessage = (...args) => translateFormattedMessageUsingPseudoLocale(formatMessage(...args));
    }
  }

  render() {
    return this.props.children;
  }

}

_defineProperty(PseudoLocaleWrapper, "propTypes", {
  children: PropTypes.element.isRequired
});

_defineProperty(PseudoLocaleWrapper, "contextTypes", {
  intl: PropTypes.object.isRequired
});