"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getConfigFromFiles = void 0;
const fs_1 = require("fs");
const js_yaml_1 = require("js-yaml");
const safer_lodash_set_1 = require("@elastic/safer-lodash-set");
const lodash_1 = require("lodash");
const ensure_deep_object_1 = require("./ensure_deep_object");
const readYaml = (path) => js_yaml_1.safeLoad(fs_1.readFileSync(path, 'utf8'));
function replaceEnvVarRefs(val) {
    return val.replace(/\$\{(\w+)\}/g, (match, envVarName) => {
        const envVarValue = process.env[envVarName];
        if (envVarValue !== undefined) {
            return envVarValue;
        }
        throw new Error(`Unknown environment variable referenced in config : ${envVarName}`);
    });
}
function merge(target, value, key) {
    if ((lodash_1.isPlainObject(value) || Array.isArray(value)) && Object.keys(value).length > 0) {
        for (const [subKey, subVal] of Object.entries(value)) {
            merge(target, subVal, key ? `${key}.${subKey}` : subKey);
        }
    }
    else if (key !== undefined) {
        safer_lodash_set_1.set(target, key, typeof value === 'string' ? replaceEnvVarRefs(value) : value);
    }
    return target;
}
/** @internal */
const getConfigFromFiles = (configFiles) => {
    let mergedYaml = {};
    for (const configFile of configFiles) {
        const yaml = readYaml(configFile);
        if (yaml !== null) {
            mergedYaml = merge(mergedYaml, yaml);
        }
    }
    return ensure_deep_object_1.ensureDeepObject(mergedYaml);
};
exports.getConfigFromFiles = getConfigFromFiles;
