"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.RawConfigService = void 0;
const tslib_1 = require("tslib");
const lodash_1 = require("lodash");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const type_detect_1 = tslib_1.__importDefault(require("type-detect"));
const read_config_1 = require("./read_config");
/** @internal */
class RawConfigService {
    constructor(configFiles, configAdapter = (rawConfig) => rawConfig) {
        this.configFiles = configFiles;
        /**
         * The stream of configs read from the config file.
         *
         * This is the _raw_ config before any overrides are applied.
         */
        this.rawConfigFromFile$ = new rxjs_1.ReplaySubject(1);
        this.config$ = this.rawConfigFromFile$.pipe(operators_1.map((rawConfig) => {
            if (lodash_1.isPlainObject(rawConfig)) {
                // TODO Make config consistent, e.g. handle dots in keys
                return configAdapter(lodash_1.cloneDeep(rawConfig));
            }
            throw new Error(`the raw config must be an object, got [${type_detect_1.default(rawConfig)}]`);
        }));
    }
    /**
     * Read the initial Kibana config.
     */
    loadConfig() {
        this.rawConfigFromFile$.next(read_config_1.getConfigFromFiles(this.configFiles));
    }
    stop() {
        this.rawConfigFromFile$.complete();
    }
    /**
     * Re-read the Kibana config.
     */
    reloadConfig() {
        this.loadConfig();
    }
    getConfig$() {
        return this.config$;
    }
}
exports.RawConfigService = RawConfigService;
