"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.configDeprecationFactory = void 0;
const lodash_1 = require("lodash");
const safer_lodash_set_1 = require("@elastic/safer-lodash-set");
const std_1 = require("@kbn/std");
const _rename = (config, rootPath, log, oldKey, newKey, silent) => {
    const fullOldPath = getPath(rootPath, oldKey);
    const oldValue = lodash_1.get(config, fullOldPath);
    if (oldValue === undefined) {
        return config;
    }
    std_1.unset(config, fullOldPath);
    const fullNewPath = getPath(rootPath, newKey);
    const newValue = lodash_1.get(config, fullNewPath);
    if (newValue === undefined) {
        safer_lodash_set_1.set(config, fullNewPath, oldValue);
        if (!silent) {
            log(`"${fullOldPath}" is deprecated and has been replaced by "${fullNewPath}"`);
        }
    }
    else {
        if (!silent) {
            log(`"${fullOldPath}" is deprecated and has been replaced by "${fullNewPath}". However both key are present, ignoring "${fullOldPath}"`);
        }
    }
    return config;
};
const _unused = (config, rootPath, log, unusedKey) => {
    const fullPath = getPath(rootPath, unusedKey);
    if (lodash_1.get(config, fullPath) === undefined) {
        return config;
    }
    std_1.unset(config, fullPath);
    log(`${fullPath} is deprecated and is no longer used`);
    return config;
};
const rename = (oldKey, newKey) => (config, rootPath, log) => _rename(config, rootPath, log, oldKey, newKey);
const renameFromRoot = (oldKey, newKey, silent) => (config, rootPath, log) => _rename(config, '', log, oldKey, newKey, silent);
const unused = (unusedKey) => (config, rootPath, log) => _unused(config, rootPath, log, unusedKey);
const unusedFromRoot = (unusedKey) => (config, rootPath, log) => _unused(config, '', log, unusedKey);
const getPath = (rootPath, subPath) => rootPath !== '' ? `${rootPath}.${subPath}` : subPath;
/**
 * The actual platform implementation of {@link ConfigDeprecationFactory}
 *
 * @internal
 */
exports.configDeprecationFactory = {
    rename,
    renameFromRoot,
    unused,
    unusedFromRoot,
};
