"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigService = void 0;
const lodash_1 = require("lodash");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const config_1 = require("./config");
const deprecation_1 = require("./deprecation");
const legacy_1 = require("./legacy");
/** @internal */
class ConfigService {
    constructor(rawConfigProvider, env, logger) {
        this.rawConfigProvider = rawConfigProvider;
        this.env = env;
        this.validated = false;
        /**
         * Whenever a config if read at a path, we mark that path as 'handled'. We can
         * then list all unhandled config paths when the startup process is completed.
         */
        this.handledPaths = new Set();
        this.schemas = new Map();
        this.deprecations = new rxjs_1.BehaviorSubject([]);
        this.log = logger.get('config');
        this.deprecationLog = logger.get('config', 'deprecation');
        this.config$ = rxjs_1.combineLatest([this.rawConfigProvider.getConfig$(), this.deprecations]).pipe(operators_1.map(([rawConfig, deprecations]) => {
            const migrated = deprecation_1.applyDeprecations(rawConfig, deprecations);
            return new legacy_1.LegacyObjectToConfigAdapter(migrated);
        }), operators_1.tap((config) => {
            this.lastConfig = config;
        }), operators_1.shareReplay(1));
    }
    /**
     * Set config schema for a path and performs its validation
     */
    setSchema(path, schema) {
        const namespace = pathToString(path);
        if (this.schemas.has(namespace)) {
            throw new Error(`Validation schema for [${path}] was already registered.`);
        }
        this.schemas.set(namespace, schema);
        this.markAsHandled(path);
    }
    /**
     * Register a {@link ConfigDeprecationProvider} to be used when validating and migrating the configuration
     */
    addDeprecationProvider(path, provider) {
        const flatPath = pathToString(path);
        this.deprecations.next([
            ...this.deprecations.value,
            ...provider(deprecation_1.configDeprecationFactory).map((deprecation) => ({
                deprecation,
                path: flatPath,
            })),
        ]);
    }
    /**
     * Validate the whole configuration and log the deprecation warnings.
     *
     * This must be done after every schemas and deprecation providers have been registered.
     */
    async validate() {
        const namespaces = [...this.schemas.keys()];
        for (let i = 0; i < namespaces.length; i++) {
            await this.getValidatedConfigAtPath$(namespaces[i]).pipe(operators_1.first()).toPromise();
        }
        await this.logDeprecation();
        this.validated = true;
    }
    /**
     * Returns the full config object observable. This is not intended for
     * "normal use", but for internal features that _need_ access to the full object.
     */
    getConfig$() {
        return this.config$;
    }
    /**
     * Reads the subset of the config at the specified `path` and validates it
     * against its registered schema.
     *
     * @param path - The path to the desired subset of the config.
     */
    atPath(path) {
        return this.getValidatedConfigAtPath$(path);
    }
    /**
     * Similar to {@link atPath}, but return the last emitted value synchronously instead of an
     * observable.
     *
     * @param path - The path to the desired subset of the config.
     */
    atPathSync(path) {
        if (!this.validated) {
            throw new Error('`atPathSync` called before config was validated');
        }
        const configAtPath = this.lastConfig.get(path);
        return this.validateAtPath(path, configAtPath);
    }
    async isEnabledAtPath(path) {
        const namespace = pathToString(path);
        const validatedConfig = this.schemas.has(namespace)
            ? await this.atPath(path).pipe(operators_1.first()).toPromise()
            : undefined;
        const enabledPath = createPluginEnabledPath(path);
        const config = await this.config$.pipe(operators_1.first()).toPromise();
        // if plugin hasn't got a config schema, we try to read "enabled" directly
        const isEnabled = validatedConfig?.enabled ?? config.get(enabledPath);
        // not declared. consider that plugin is enabled by default
        if (isEnabled === undefined) {
            return true;
        }
        if (isEnabled === false) {
            // If the plugin is _not_ enabled, we mark the entire plugin path as
            // handled, as it's expected that it won't be used.
            this.markAsHandled(path);
            return false;
        }
        // If plugin enabled we mark the enabled path as handled, as we for example
        // can have plugins that don't have _any_ config except for this field, and
        // therefore have no reason to try to get the config.
        this.markAsHandled(enabledPath);
        return true;
    }
    async getUnusedPaths() {
        const config = await this.config$.pipe(operators_1.first()).toPromise();
        const handledPaths = [...this.handledPaths.values()].map(pathToString);
        return config.getFlattenedPaths().filter((path) => !isPathHandled(path, handledPaths));
    }
    async getUsedPaths() {
        const config = await this.config$.pipe(operators_1.first()).toPromise();
        const handledPaths = [...this.handledPaths.values()].map(pathToString);
        return config.getFlattenedPaths().filter((path) => isPathHandled(path, handledPaths));
    }
    async logDeprecation() {
        const rawConfig = await this.rawConfigProvider.getConfig$().pipe(operators_1.take(1)).toPromise();
        const deprecations = await this.deprecations.pipe(operators_1.take(1)).toPromise();
        const deprecationMessages = [];
        const logger = (msg) => deprecationMessages.push(msg);
        deprecation_1.applyDeprecations(rawConfig, deprecations, logger);
        deprecationMessages.forEach((msg) => {
            this.deprecationLog.warn(msg);
        });
    }
    validateAtPath(path, config) {
        const namespace = pathToString(path);
        const schema = this.schemas.get(namespace);
        if (!schema) {
            throw new Error(`No validation schema has been defined for [${namespace}]`);
        }
        return schema.validate(config, {
            dev: this.env.mode.dev,
            prod: this.env.mode.prod,
            ...this.env.packageInfo,
        }, `config validation of [${namespace}]`);
    }
    getValidatedConfigAtPath$(path) {
        return this.config$.pipe(operators_1.map((config) => config.get(path)), operators_1.distinctUntilChanged(lodash_1.isEqual), operators_1.map((config) => this.validateAtPath(path, config)));
    }
    markAsHandled(path) {
        this.log.debug(`Marking config path as handled: ${path}`);
        this.handledPaths.add(path);
    }
}
exports.ConfigService = ConfigService;
const createPluginEnabledPath = (configPath) => {
    if (Array.isArray(configPath)) {
        return configPath.concat('enabled');
    }
    return `${configPath}.enabled`;
};
const pathToString = (path) => (Array.isArray(path) ? path.join('.') : path);
/**
 * A path is considered 'handled' if it is a subset of any of the already
 * handled paths.
 */
const isPathHandled = (path, handledPaths) => handledPaths.some((handledPath) => config_1.hasConfigPathIntersection(path, handledPath));
