"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringType = void 0;
const tslib_1 = require("tslib");
const type_detect_1 = tslib_1.__importDefault(require("type-detect"));
const internals_1 = require("../internals");
const type_1 = require("./type");
class StringType extends type_1.Type {
    constructor(options = {}) {
        // We want to allow empty strings, however calling `allow('')` causes
        // Joi to allow the value and skip any additional validation.
        // Instead, we reimplement the string validator manually except in the
        // hostname case where empty strings aren't allowed anyways.
        let schema = options.hostname === true
            ? internals_1.internals.string().hostname()
            : internals_1.internals.any().custom((value) => {
                if (typeof value !== 'string') {
                    return `expected value of type [string] but got [${type_detect_1.default(value)}]`;
                }
            });
        if (options.minLength !== undefined) {
            schema = schema.custom((value) => {
                if (value.length < options.minLength) {
                    return `value has length [${value.length}] but it must have a minimum length of [${options.minLength}].`;
                }
            });
        }
        if (options.maxLength !== undefined) {
            schema = schema.custom((value) => {
                if (value.length > options.maxLength) {
                    return `value has length [${value.length}] but it must have a maximum length of [${options.maxLength}].`;
                }
            });
        }
        super(schema, options);
    }
    handleError(type, { limit, value }) {
        switch (type) {
            case 'any.required':
                return `expected value of type [string] but got [${type_detect_1.default(value)}]`;
            case 'string.hostname':
                return `value must be a valid hostname (see RFC 1123).`;
        }
    }
}
exports.StringType = StringType;
