"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.withSpan = exports.parseSpanOptions = void 0;
const tslib_1 = require("tslib");
const elastic_apm_node_1 = tslib_1.__importDefault(require("elastic-apm-node"));
const async_hooks_1 = tslib_1.__importDefault(require("async_hooks"));
function parseSpanOptions(optionsOrName) {
    const options = typeof optionsOrName === 'string' ? { name: optionsOrName } : optionsOrName;
    return options;
}
exports.parseSpanOptions = parseSpanOptions;
const runInNewContext = (cb) => {
    const resource = new async_hooks_1.default.AsyncResource('fake_async');
    return resource.runInAsyncScope(cb);
};
async function withSpan(optionsOrName, cb) {
    const options = parseSpanOptions(optionsOrName);
    const { name, type, subtype, labels } = options;
    if (!elastic_apm_node_1.default.isStarted()) {
        return cb();
    }
    // When a span starts, it's marked as the active span in its context.
    // When it ends, it's not untracked, which means that if a span
    // starts directly after this one ends, the newly started span is a
    // child of this span, even though it should be a sibling.
    // To mitigate this, we queue a microtask by awaiting a promise.
    await Promise.resolve();
    const span = elastic_apm_node_1.default.startSpan(name);
    if (!span) {
        return cb();
    }
    // If a span is created in the same context as the span that we just
    // started, it will be a sibling, not a child. E.g., the Elasticsearch span
    // that is created when calling search() happens in the same context. To
    // mitigate this we create a new context.
    return runInNewContext(() => {
        // @ts-ignore
        if (type) {
            span.type = type;
        }
        if (subtype) {
            span.subtype = subtype;
        }
        if (labels) {
            span.addLabels(labels);
        }
        return cb()
            .then((res) => {
            span.outcome = 'success';
            return res;
        })
            .catch((err) => {
            span.outcome = 'failure';
            throw err;
        })
            .finally(() => {
            span.end();
        });
    });
}
exports.withSpan = withSpan;
