"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApmConfiguration = void 0;
const path_1 = require("path");
const lodash_1 = require("lodash");
const child_process_1 = require("child_process");
// deep import to avoid loading the whole package
const path_2 = require("@kbn/utils/target/path");
const fs_1 = require("fs");
const getDefaultConfig = (isDistributable) => {
    // https://www.elastic.co/guide/en/apm/agent/nodejs/current/configuration.html
    return {
        active: process.env.ELASTIC_APM_ACTIVE === 'true' || false,
        environment: process.env.ELASTIC_APM_ENVIRONMENT || process.env.NODE_ENV || 'development',
        serverUrl: 'https://38b80fbd79fb4c91bae06b4642d4d093.apm.us-east-1.aws.cloud.es.io',
        // The secretToken below is intended to be hardcoded in this file even though
        // it makes it public. This is not a security/privacy issue. Normally we'd
        // instead disable the need for a secretToken in the APM Server config where
        // the data is transmitted to, but due to how it's being hosted, it's easier,
        // for now, to simply leave it in.
        secretToken: 'ZQHYvrmXEx04ozge8F',
        logUncaughtExceptions: true,
        globalLabels: {},
        centralConfig: false,
        metricsInterval: isDistributable ? '120s' : '30s',
        captureSpanStackTraces: false,
        transactionSampleRate: process.env.ELASTIC_APM_TRANSACTION_SAMPLE_RATE
            ? parseFloat(process.env.ELASTIC_APM_TRANSACTION_SAMPLE_RATE)
            : 1.0,
        // Can be performance intensive, disabling by default
        breakdownMetrics: isDistributable ? false : true,
    };
};
class ApmConfiguration {
    constructor(rootDir, rawKibanaConfig, isDistributable) {
        this.rootDir = rootDir;
        this.rawKibanaConfig = rawKibanaConfig;
        this.isDistributable = isDistributable;
        // eslint-disable-next-line @typescript-eslint/no-var-requires
        const { version, build } = require(path_1.join(this.rootDir, 'package.json'));
        this.kibanaVersion = version;
        this.pkgBuild = build;
    }
    getConfig(serviceName) {
        return {
            ...this.getBaseConfig(),
            serviceName,
        };
    }
    getBaseConfig() {
        if (!this.baseConfig) {
            const apmConfig = lodash_1.merge(getDefaultConfig(this.isDistributable), this.getConfigFromKibanaConfig(), this.getDevConfig(), this.getDistConfig(), this.getCIConfig());
            const rev = this.getGitRev();
            if (rev !== null) {
                apmConfig.globalLabels.git_rev = rev;
            }
            const uuid = this.getKibanaUuid();
            if (uuid) {
                apmConfig.globalLabels.kibana_uuid = uuid;
            }
            apmConfig.serviceVersion = this.kibanaVersion;
            this.baseConfig = apmConfig;
        }
        return this.baseConfig;
    }
    getConfigFromKibanaConfig() {
        return lodash_1.get(this.rawKibanaConfig, 'elastic.apm', {});
    }
    getKibanaUuid() {
        // try to access the `server.uuid` value from the config file first.
        // if not manually defined, we will then read the value from the `{DATA_FOLDER}/uuid` file.
        // note that as the file is created by the platform AFTER apm init, the file
        // will not be present at first startup, but there is nothing we can really do about that.
        if (lodash_1.get(this.rawKibanaConfig, 'server.uuid')) {
            return this.rawKibanaConfig.server.uuid;
        }
        const dataPath = lodash_1.get(this.rawKibanaConfig, 'path.data') || path_2.getDataPath();
        try {
            const filename = path_1.join(dataPath, 'uuid');
            return fs_1.readFileSync(filename, 'utf-8');
        }
        catch (e) { } // eslint-disable-line no-empty
    }
    getDevConfig() {
        try {
            const apmDevConfigPath = path_1.join(this.rootDir, 'config', 'apm.dev.js');
            return require(apmDevConfigPath);
        }
        catch (e) {
            return {};
        }
    }
    /** Config keys that cannot be overridden in production builds */
    getDistConfig() {
        if (!this.isDistributable) {
            return {};
        }
        return {
            // Headers & body may contain sensitive info
            captureHeaders: false,
            captureBody: 'off',
        };
    }
    getCIConfig() {
        if (process.env.ELASTIC_APM_ENVIRONMENT !== 'ci') {
            return {};
        }
        return {
            globalLabels: {
                branch: process.env.GIT_BRANCH || '',
                targetBranch: process.env.PR_TARGET_BRANCH || '',
                ciBuildNumber: process.env.BUILD_NUMBER || '',
                isPr: process.env.GITHUB_PR_NUMBER ? true : false,
                prId: process.env.GITHUB_PR_NUMBER || '',
            },
        };
    }
    getGitRev() {
        if (this.isDistributable) {
            return this.pkgBuild.sha;
        }
        try {
            return child_process_1.execSync('git rev-parse --short HEAD', {
                encoding: 'utf-8',
                stdio: ['ignore', 'pipe', 'ignore'],
            }).trim();
        }
        catch (e) {
            return null;
        }
    }
}
exports.ApmConfiguration = ApmConfiguration;
