function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import { wrapArray } from './util';
import { createUiCounterMetric, trackUsageAgent } from './metrics';
import { ReportStorageManager } from './storage';
import { ReportManager } from './report';
export class Reporter {
  constructor(config) {
    _defineProperty(this, "checkInterval", void 0);

    _defineProperty(this, "interval", void 0);

    _defineProperty(this, "http", void 0);

    _defineProperty(this, "reportManager", void 0);

    _defineProperty(this, "storageManager", void 0);

    _defineProperty(this, "debug", void 0);

    _defineProperty(this, "retryCount", 0);

    _defineProperty(this, "maxRetries", 3);

    _defineProperty(this, "start", () => {
      if (!this.interval) {
        this.interval = setTimeout(() => {
          this.interval = undefined;
          this.sendReports();
        }, this.checkInterval);
      }
    });

    _defineProperty(this, "reportUiCounter", (appName, type, eventNames, count) => {
      const metrics = wrapArray(eventNames).map(eventName => {
        this.log(`${type} Metric -> (${appName}:${eventName}):`);
        const report = createUiCounterMetric({
          type,
          appName,
          eventName,
          count
        });
        this.log(report);
        return report;
      });
      this.saveToReport(metrics);
    });

    _defineProperty(this, "reportUserAgent", appName => {
      this.log(`Reporting user-agent.`);
      const report = trackUsageAgent(appName);
      this.saveToReport([report]);
    });

    _defineProperty(this, "sendReports", async () => {
      if (!this.reportManager.isReportEmpty()) {
        try {
          await this.http(this.reportManager.report);
          this.flushReport();
        } catch (err) {
          this.log(`Error Sending Metrics Report ${err}`);
          this.retryCount = this.retryCount + 1;
          const versionMismatch = this.reportManager.report.reportVersion !== ReportManager.REPORT_VERSION;

          if (versionMismatch || this.retryCount > this.maxRetries) {
            this.flushReport();
          }
        }
      }

      this.start();
    });

    const {
      http,
      storage,
      debug,
      checkInterval = 90000,
      storageKey = 'analytics'
    } = config;
    this.http = http;
    this.checkInterval = checkInterval;
    this.storageManager = new ReportStorageManager(storageKey, storage);
    const storedReport = this.storageManager.get();
    this.reportManager = new ReportManager(storedReport);
    this.debug = !!debug;
  }

  saveToReport(newMetrics) {
    this.reportManager.assignReports(newMetrics);
    this.storageManager.store(this.reportManager.report);
  }

  flushReport() {
    this.retryCount = 0;
    this.reportManager.clearReport();
    this.storageManager.store(this.reportManager.report);
  }

  log(message) {
    if (this.debug) {
      // eslint-disable-next-line
      console.debug(message);
    }
  }

  reportApplicationUsage(appUsageReport) {
    this.log(`Reporting application usage for ${appUsageReport.appId}, ${appUsageReport.viewId}`);
    this.saveToReport([appUsageReport]);
  }

}