function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
import moment from 'moment-timezone';
import { UnreachableCaseError, wrapArray } from './util';
import { ApplicationUsageTracker } from './application_usage_tracker';
import { METRIC_TYPE } from './metrics';
const REPORT_VERSION = 3;
export class ReportManager {
  constructor(report) {
    _defineProperty(this, "report", void 0);

    this.report = report || ReportManager.createReport();
  }

  static createReport() {
    return {
      reportVersion: REPORT_VERSION
    };
  }

  clearReport() {
    this.report = ReportManager.createReport();
  }

  isReportEmpty() {
    const {
      uiCounter,
      userAgent,
      application_usage: appUsage
    } = this.report;
    const noUiCounters = !uiCounter || Object.keys(uiCounter).length === 0;
    const noUserAgents = !userAgent || Object.keys(userAgent).length === 0;
    const noAppUsage = !appUsage || Object.keys(appUsage).length === 0;
    return noUiCounters && noUserAgents && noAppUsage;
  }

  incrementTotal(count, currentTotal) {
    const currentTotalNumber = typeof currentTotal === 'number' ? currentTotal : 0;
    return count + currentTotalNumber;
  }

  assignReports(newMetrics) {
    wrapArray(newMetrics).forEach(newMetric => this.assignReport(this.report, newMetric));
    return {
      report: this.report
    };
  }

  static createMetricKey(metric) {
    switch (metric.type) {
      case METRIC_TYPE.USER_AGENT:
        {
          const {
            appName,
            type
          } = metric;
          return `${appName}-${type}`;
        }

      case METRIC_TYPE.CLICK:
      case METRIC_TYPE.LOADED:
      case METRIC_TYPE.COUNT:
        {
          const {
            appName,
            eventName,
            type
          } = metric;
          return `${appName}-${type}-${eventName}`;
        }

      case METRIC_TYPE.APPLICATION_USAGE:
        {
          const {
            appId,
            viewId
          } = metric;
          return ApplicationUsageTracker.serializeKey({
            appId,
            viewId
          });
        }

      default:
        throw new UnreachableCaseError(metric);
    }
  }

  assignReport(report, metric) {
    const key = ReportManager.createMetricKey(metric);

    switch (metric.type) {
      case METRIC_TYPE.USER_AGENT:
        {
          const {
            appName,
            type,
            userAgent
          } = metric;

          if (userAgent) {
            report.userAgent = {
              [key]: {
                key,
                appName,
                type,
                userAgent: metric.userAgent
              }
            };
          }

          return;
        }

      case METRIC_TYPE.CLICK:
      case METRIC_TYPE.LOADED:
      case METRIC_TYPE.COUNT:
        {
          var _report$uiCounter$key;

          const {
            appName,
            type,
            eventName,
            count
          } = metric;
          report.uiCounter = report.uiCounter || {};
          const currentTotal = (_report$uiCounter$key = report.uiCounter[key]) === null || _report$uiCounter$key === void 0 ? void 0 : _report$uiCounter$key.total;
          report.uiCounter[key] = {
            key,
            appName,
            eventName,
            type,
            total: this.incrementTotal(count, currentTotal)
          };
          return;
        }

      case METRIC_TYPE.APPLICATION_USAGE:
        {
          const {
            numberOfClicks,
            startTime,
            appId,
            viewId
          } = metric;
          const minutesOnScreen = moment().diff(startTime, 'minutes', true);
          report.application_usage = report.application_usage || {};
          const appExistingData = report.application_usage[key] || {
            minutesOnScreen: 0,
            numberOfClicks: 0,
            appId,
            viewId
          };
          report.application_usage[key] = { ...appExistingData,
            minutesOnScreen: appExistingData.minutesOnScreen + minutesOnScreen,
            numberOfClicks: appExistingData.numberOfClicks + numberOfClicks
          };
          return;
        }

      default:
        throw new UnreachableCaseError(metric);
    }
  }

}

_defineProperty(ReportManager, "REPORT_VERSION", REPORT_VERSION);