"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReportManager = void 0;

var _momentTimezone = _interopRequireDefault(require("moment-timezone"));

var _util = require("./util");

var _application_usage_tracker = require("./application_usage_tracker");

var _metrics = require("./metrics");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const REPORT_VERSION = 3;

class ReportManager {
  constructor(report) {
    _defineProperty(this, "report", void 0);

    this.report = report || ReportManager.createReport();
  }

  static createReport() {
    return {
      reportVersion: REPORT_VERSION
    };
  }

  clearReport() {
    this.report = ReportManager.createReport();
  }

  isReportEmpty() {
    const {
      uiCounter,
      userAgent,
      application_usage: appUsage
    } = this.report;
    const noUiCounters = !uiCounter || Object.keys(uiCounter).length === 0;
    const noUserAgents = !userAgent || Object.keys(userAgent).length === 0;
    const noAppUsage = !appUsage || Object.keys(appUsage).length === 0;
    return noUiCounters && noUserAgents && noAppUsage;
  }

  incrementTotal(count, currentTotal) {
    const currentTotalNumber = typeof currentTotal === 'number' ? currentTotal : 0;
    return count + currentTotalNumber;
  }

  assignReports(newMetrics) {
    (0, _util.wrapArray)(newMetrics).forEach(newMetric => this.assignReport(this.report, newMetric));
    return {
      report: this.report
    };
  }

  static createMetricKey(metric) {
    switch (metric.type) {
      case _metrics.METRIC_TYPE.USER_AGENT:
        {
          const {
            appName,
            type
          } = metric;
          return `${appName}-${type}`;
        }

      case _metrics.METRIC_TYPE.CLICK:
      case _metrics.METRIC_TYPE.LOADED:
      case _metrics.METRIC_TYPE.COUNT:
        {
          const {
            appName,
            eventName,
            type
          } = metric;
          return `${appName}-${type}-${eventName}`;
        }

      case _metrics.METRIC_TYPE.APPLICATION_USAGE:
        {
          const {
            appId,
            viewId
          } = metric;
          return _application_usage_tracker.ApplicationUsageTracker.serializeKey({
            appId,
            viewId
          });
        }

      default:
        throw new _util.UnreachableCaseError(metric);
    }
  }

  assignReport(report, metric) {
    const key = ReportManager.createMetricKey(metric);

    switch (metric.type) {
      case _metrics.METRIC_TYPE.USER_AGENT:
        {
          const {
            appName,
            type,
            userAgent
          } = metric;

          if (userAgent) {
            report.userAgent = {
              [key]: {
                key,
                appName,
                type,
                userAgent: metric.userAgent
              }
            };
          }

          return;
        }

      case _metrics.METRIC_TYPE.CLICK:
      case _metrics.METRIC_TYPE.LOADED:
      case _metrics.METRIC_TYPE.COUNT:
        {
          var _report$uiCounter$key;

          const {
            appName,
            type,
            eventName,
            count
          } = metric;
          report.uiCounter = report.uiCounter || {};
          const currentTotal = (_report$uiCounter$key = report.uiCounter[key]) === null || _report$uiCounter$key === void 0 ? void 0 : _report$uiCounter$key.total;
          report.uiCounter[key] = {
            key,
            appName,
            eventName,
            type,
            total: this.incrementTotal(count, currentTotal)
          };
          return;
        }

      case _metrics.METRIC_TYPE.APPLICATION_USAGE:
        {
          const {
            numberOfClicks,
            startTime,
            appId,
            viewId
          } = metric;
          const minutesOnScreen = (0, _momentTimezone.default)().diff(startTime, 'minutes', true);
          report.application_usage = report.application_usage || {};
          const appExistingData = report.application_usage[key] || {
            minutesOnScreen: 0,
            numberOfClicks: 0,
            appId,
            viewId
          };
          report.application_usage[key] = { ...appExistingData,
            minutesOnScreen: appExistingData.minutesOnScreen + minutesOnScreen,
            numberOfClicks: appExistingData.numberOfClicks + numberOfClicks
          };
          return;
        }

      default:
        throw new _util.UnreachableCaseError(metric);
    }
  }

}

exports.ReportManager = ReportManager;

_defineProperty(ReportManager, "REPORT_VERSION", REPORT_VERSION);