"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = require("fs");
const js_yaml_1 = require("js-yaml");
const path_1 = require("path");
// look for telemetry.yml in the same places we expect kibana.yml
const ensure_deep_object_1 = require("./ensure_deep_object");
/**
 * The maximum file size before we ignore it (note: this limit is arbitrary).
 */
exports.MAX_FILE_SIZE = 10 * 1024; // 10 KB
/**
 * Determine if the supplied `path` is readable.
 *
 * @param path The possible path where a config file may exist.
 * @returns `true` if the file should be used.
 */
function isFileReadable(path) {
    try {
        fs_1.accessSync(path, fs_1.constants.R_OK);
        // ignore files above the limit
        const stats = fs_1.statSync(path);
        return stats.size <= exports.MAX_FILE_SIZE;
    }
    catch (e) {
        return false;
    }
}
exports.isFileReadable = isFileReadable;
/**
 * Load the `telemetry.yml` file, if it exists, and return its contents as
 * a JSON object.
 *
 * @param configPath The config file path.
 * @returns The unmodified JSON object if the file exists and is a valid YAML file.
 */
async function readTelemetryFile(path) {
    try {
        if (isFileReadable(path)) {
            const yaml = fs_1.readFileSync(path);
            const data = js_yaml_1.safeLoad(yaml.toString());
            // don't bother returning empty objects
            if (Object.keys(data).length) {
                // ensure { "a.b": "value" } becomes { "a": { "b": "value" } }
                return ensure_deep_object_1.ensureDeepObject(data);
            }
        }
    }
    catch (e) {
        // ignored
    }
    return undefined;
}
exports.readTelemetryFile = readTelemetryFile;
/**
 * Create a usage collector that provides the `telemetry.yml` data as a
 * `static_telemetry` object.
 *
 * Loading of the file is done lazily and on-demand. This avoids hanging
 * onto the data in memory unnecessarily, as well as allows it to be
 * updated out-of-process without having to restart Kibana.
 *
 * @param server The Kibana server instance.
 * @return `UsageCollector` that provides the `static_telemetry` described.
 */
function createTelemetryUsageCollector(server) {
    return server.usage.collectorSet.makeUsageCollector({
        type: 'static_telemetry',
        isReady: () => true,
        fetch: async () => {
            const configPath = server.config().get('xpack.xpack_main.telemetry.config');
            const telemetryPath = path_1.join(path_1.dirname(configPath), 'telemetry.yml');
            return await readTelemetryFile(telemetryPath);
        },
    });
}
exports.createTelemetryUsageCollector = createTelemetryUsageCollector;
