"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const lodash_1 = require("lodash");
const moment_1 = tslib_1.__importDefault(require("moment"));
const constants_1 = require("../../../../common/constants");
const helper_1 = require("../../helper");
class ElasticsearchPingsAdapter {
    constructor(database) {
        this.database = database;
    }
    async getAll(request, dateRangeStart, dateRangeEnd, monitorId, status, sort, size) {
        const sortParam = sort ? { sort: [{ '@timestamp': { order: sort } }] } : undefined;
        const sizeParam = size ? { size } : undefined;
        const must = [];
        if (monitorId) {
            must.push({ term: { 'monitor.id': monitorId } });
        }
        if (status) {
            must.push({ term: { 'monitor.status': status } });
        }
        const filter = [{ range: { '@timestamp': { gte: dateRangeStart, lte: dateRangeEnd } } }];
        const queryContext = { bool: { must, filter } };
        const params = {
            index: constants_1.INDEX_NAMES.HEARTBEAT,
            rest_total_hits_as_int: true,
            body: {
                query: {
                    ...queryContext,
                },
                ...sortParam,
                ...sizeParam,
            },
        };
        const { hits: { hits, total }, } = await this.database.search(request, params);
        const pings = hits.map(({ _source }) => {
            const timestamp = _source['@timestamp'];
            return { timestamp, ..._source };
        });
        const results = {
            total,
            pings,
        };
        return results;
    }
    async getLatestMonitorDocs(request, dateRangeStart, dateRangeEnd, monitorId) {
        const must = [];
        if (monitorId) {
            must.push({ term: { 'monitor.id': monitorId } });
        }
        const params = {
            index: constants_1.INDEX_NAMES.HEARTBEAT,
            body: {
                query: {
                    bool: {
                        must: must.length ? [...must] : undefined,
                        filter: [
                            {
                                range: {
                                    '@timestamp': {
                                        gte: dateRangeStart,
                                        lte: dateRangeEnd,
                                    },
                                },
                            },
                        ],
                    },
                },
                aggs: {
                    by_id: {
                        terms: {
                            field: 'monitor.id',
                        },
                        aggs: {
                            latest: {
                                top_hits: {
                                    size: 1,
                                    sort: {
                                        '@timestamp': { order: 'desc' },
                                    },
                                },
                            },
                        },
                    },
                },
            },
        };
        const { aggregations: { by_id: { buckets }, }, } = await this.database.search(request, params);
        // @ts-ignore TODO fix destructuring implicit any
        return buckets.map(({ latest: { hits: { hits } } }) => {
            const timestamp = hits[0]._source[`@timestamp`];
            const momentTs = moment_1.default(timestamp);
            const millisFromNow = moment_1.default().diff(momentTs);
            return {
                ...hits[0]._source,
                timestamp,
                millisFromNow,
            };
        });
    }
    async getPingHistogram(request, dateRangeStart, dateRangeEnd, filters) {
        const query = helper_1.getFilterFromMust(dateRangeStart, dateRangeEnd, filters);
        const params = {
            index: constants_1.INDEX_NAMES.HEARTBEAT,
            rest_total_hits_as_int: true,
            body: {
                query,
                size: 0,
                aggs: {
                    timeseries: {
                        auto_date_histogram: {
                            field: '@timestamp',
                            buckets: 50,
                        },
                        aggs: {
                            down: {
                                filter: {
                                    term: {
                                        'monitor.status': 'down',
                                    },
                                },
                                aggs: {
                                    bucket_count: {
                                        cardinality: {
                                            field: 'monitor.id',
                                        },
                                    },
                                },
                            },
                            bucket_total: {
                                cardinality: {
                                    field: 'monitor.id',
                                    precision_threshold: 20000,
                                },
                            },
                        },
                    },
                },
            },
        };
        const result = await this.database.search(request, params);
        const buckets = lodash_1.get(result, 'aggregations.timeseries.buckets', []);
        const mappedBuckets = buckets.map(bucket => {
            const key = lodash_1.get(bucket, 'key');
            const total = lodash_1.get(bucket, 'bucket_total.value');
            const downCount = lodash_1.get(bucket, 'down.bucket_count.value');
            return {
                key,
                downCount,
                upCount: total - downCount,
                y: 1,
            };
        });
        return helper_1.formatEsBucketsForHistogram(mappedBuckets);
    }
    async getDocCount(request) {
        const { count } = await this.database.count(request, { index: constants_1.INDEX_NAMES.HEARTBEAT });
        return { count };
    }
}
exports.ElasticsearchPingsAdapter = ElasticsearchPingsAdapter;
