"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lodash_1 = require("lodash");
const constants_1 = require("../../../../common/constants");
const helper_1 = require("../../helper");
// the values for these charts are stored as μs, but should be displayed as ms
const formatChartValue = (time, chartPoint) => ({
    x: time,
    y: chartPoint.value === null ? null : chartPoint.value / 1000,
});
const formatStatusBuckets = (time, buckets, docCount) => {
    let up = null;
    let down = null;
    buckets.forEach((bucket) => {
        if (bucket.key === 'up') {
            up = bucket.doc_count;
        }
        else if (bucket.key === 'down') {
            down = bucket.doc_count;
        }
    });
    return {
        x: time,
        up,
        down,
        total: docCount,
    };
};
const getFilteredQuery = (dateRangeStart, dateRangeEnd, filters) => {
    let filtersObj;
    // TODO: handle bad JSON gracefully
    filtersObj = filters ? JSON.parse(filters) : undefined;
    const query = { ...filtersObj };
    const rangeSection = {
        range: {
            '@timestamp': {
                gte: dateRangeStart,
                lte: dateRangeEnd,
            },
        },
    };
    if (lodash_1.get(query, 'bool.must', undefined)) {
        query.bool.must.push({
            ...rangeSection,
        });
    }
    else {
        lodash_1.set(query, 'bool.must', [rangeSection]);
    }
    return query;
};
class ElasticsearchMonitorsAdapter {
    constructor(database) {
        this.database = database;
        this.database = database;
    }
    async getMonitorChartsData(request, monitorId, dateRangeStart, dateRangeEnd) {
        const query = {
            bool: {
                must: [{ term: { 'monitor.id': monitorId } }],
                filter: [{ range: { '@timestamp': { gte: dateRangeStart, lte: dateRangeEnd } } }],
            },
        };
        const aggs = {
            timeseries: {
                auto_date_histogram: {
                    field: '@timestamp',
                    buckets: 50,
                },
                aggs: {
                    max_content: { max: { field: 'http.rtt.content.us' } },
                    max_response: { max: { field: 'http.rtt.response_header.us' } },
                    max_validate: { max: { field: 'http.rtt.validate.us' } },
                    max_total: { max: { field: 'http.rtt.total.us' } },
                    max_write_request: { max: { field: 'http.rtt.write_request.us' } },
                    max_tcp_rtt: { max: { field: 'tcp.rtt.connect.us' } },
                    status: { terms: { field: 'monitor.status' } },
                    max_duration: { max: { field: 'monitor.duration.us' } },
                    min_duration: { min: { field: 'monitor.duration.us' } },
                    avg_duration: { avg: { field: 'monitor.duration.us' } },
                },
            },
        };
        const params = {
            index: constants_1.INDEX_NAMES.HEARTBEAT,
            body: { query, aggs },
        };
        const { aggregations: { timeseries: { buckets }, }, } = await this.database.search(request, params);
        return buckets.map(({ key, max_content, avg_duration, max_write_request, max_validate, max_tcp_rtt, max_response, min_duration, max_total, max_duration, status, doc_count, }) => {
            return {
                maxContent: formatChartValue(key, max_content),
                avgDuration: formatChartValue(key, avg_duration),
                maxWriteRequest: formatChartValue(key, max_write_request),
                maxValidate: formatChartValue(key, max_validate),
                maxTcpRtt: formatChartValue(key, max_tcp_rtt),
                maxResponse: formatChartValue(key, max_response),
                minDuration: formatChartValue(key, min_duration),
                maxTotal: formatChartValue(key, max_total),
                maxDuration: formatChartValue(key, max_duration),
                status: formatStatusBuckets(key, status.buckets, doc_count),
            };
        });
    }
    async getSnapshotCount(request, dateRangeStart, dateRangeEnd, filter) {
        const { statusFilter, query } = helper_1.getFilteredQueryAndStatusFilter(dateRangeStart, dateRangeEnd, filter);
        const params = {
            index: constants_1.INDEX_NAMES.HEARTBEAT,
            body: {
                query,
                size: 0,
                aggs: {
                    ids: {
                        composite: {
                            sources: [
                                {
                                    id: {
                                        terms: {
                                            field: 'monitor.id',
                                        },
                                    },
                                },
                            ],
                            size: 10000,
                        },
                        aggs: {
                            latest: {
                                top_hits: {
                                    sort: [
                                        {
                                            '@timestamp': { order: 'desc' },
                                        },
                                    ],
                                    size: 1,
                                },
                            },
                        },
                    },
                },
            },
        };
        let up = 0;
        let down = 0;
        let searchAfter = null;
        do {
            if (searchAfter) {
                lodash_1.set(params, 'body.aggs.ids.composite.after', searchAfter);
            }
            const queryResult = await this.database.search(request, params);
            const idBuckets = lodash_1.get(queryResult, 'aggregations.ids.buckets', []);
            idBuckets.forEach(bucket => {
                // We only get the latest doc
                const status = lodash_1.get(bucket, 'latest.hits.hits[0]._source.monitor.status', null);
                if (!statusFilter || (statusFilter && statusFilter === status)) {
                    if (status === 'up') {
                        up++;
                    }
                    else {
                        down++;
                    }
                }
            });
            searchAfter = lodash_1.get(queryResult, 'aggregations.ids.after_key');
        } while (searchAfter);
        return { up, down, total: up + down };
    }
    async getLatestMonitors(request, dateRangeStart, dateRangeEnd, filters) {
        const { statusFilter, query } = helper_1.getFilteredQueryAndStatusFilter(dateRangeStart, dateRangeEnd, filters);
        const params = {
            index: constants_1.INDEX_NAMES.HEARTBEAT,
            body: {
                size: 0,
                query,
                aggs: {
                    hosts: {
                        composite: {
                            sources: [
                                {
                                    id: {
                                        terms: {
                                            field: 'monitor.id',
                                        },
                                    },
                                },
                                {
                                    port: {
                                        terms: {
                                            field: 'tcp.port',
                                            missing_bucket: true,
                                        },
                                    },
                                },
                            ],
                            size: 50,
                        },
                        aggs: {
                            latest: {
                                top_hits: {
                                    sort: [
                                        {
                                            '@timestamp': { order: 'desc' },
                                        },
                                    ],
                                    size: 1,
                                },
                            },
                            histogram: {
                                auto_date_histogram: {
                                    field: '@timestamp',
                                    buckets: 25,
                                },
                                aggs: {
                                    status: {
                                        terms: {
                                            field: 'monitor.status',
                                            size: 10,
                                        },
                                    },
                                },
                            },
                        },
                    },
                },
            },
        };
        const httpTcpResult = await this.database.search(request, params);
        const result = lodash_1.get(httpTcpResult, 'aggregations.hosts.buckets', [])
            .map((resultBucket) => {
            const key = lodash_1.get(resultBucket, 'key');
            const buckets = lodash_1.get(resultBucket, 'histogram.buckets', []);
            const hits = lodash_1.get(resultBucket, 'latest.hits.hits', []);
            const latestStatus = lodash_1.get(hits, '[0]._source.monitor.status', undefined);
            if (statusFilter && latestStatus !== statusFilter) {
                return undefined;
            }
            const upSeries = [];
            const downSeries = [];
            // @ts-ignore TODO update typings and remove this comment
            buckets.forEach(bucket => {
                const status = lodash_1.get(bucket, 'status.buckets', []);
                // @ts-ignore TODO update typings and remove this comment
                const up = status.find(f => f.key === 'up');
                // @ts-ignore TODO update typings and remove this comment
                const down = status.find(f => f.key === 'down');
                // @ts-ignore TODO update typings and remove this comment
                upSeries.push({ x: bucket.key, y: up ? up.doc_count : null });
                // @ts-ignore TODO update typings and remove this comment
                downSeries.push({ x: bucket.key, y: down ? down.doc_count : null });
            });
            return {
                key,
                ping: {
                    ...hits[0]._source,
                    timestamp: hits[0]._source['@timestamp'],
                },
                upSeries,
                downSeries,
            };
        })
            .filter((f) => f !== undefined);
        return result;
    }
    async getFilterBar(request, dateRangeStart, dateRangeEnd) {
        const MONITOR_SOURCE_ID_KEY = 'monitor.id';
        const MONITOR_SOURCE_TCP_KEY = 'tcp.port';
        const MONITOR_SOURCE_TYPE_KEY = 'monitor.type';
        const params = {
            index: constants_1.INDEX_NAMES.HEARTBEAT,
            body: {
                _source: [MONITOR_SOURCE_ID_KEY, MONITOR_SOURCE_TCP_KEY, MONITOR_SOURCE_TYPE_KEY],
                size: 1000,
                query: {
                    range: {
                        '@timestamp': {
                            gte: dateRangeStart,
                            lte: dateRangeEnd,
                        },
                    },
                },
                collapse: {
                    field: 'monitor.id',
                },
                sort: {
                    '@timestamp': 'desc',
                },
            },
        };
        const result = await this.database.search(request, params);
        const ids = [];
        const ports = new Set();
        const types = new Set();
        const hits = lodash_1.get(result, 'hits.hits', []);
        hits.forEach((hit) => {
            const key = lodash_1.get(hit, `_source.${MONITOR_SOURCE_ID_KEY}`);
            const portValue = lodash_1.get(hit, `_source.${MONITOR_SOURCE_TCP_KEY}`, undefined);
            const typeValue = lodash_1.get(hit, `_source.${MONITOR_SOURCE_TYPE_KEY}`, undefined);
            if (key) {
                ids.push(key);
            }
            if (portValue) {
                ports.add(portValue);
            }
            if (typeValue) {
                types.add(typeValue);
            }
        });
        return {
            type: Array.from(types).sort(),
            port: Array.from(ports).sort((a, b) => a - b),
            id: ids.sort(),
            status: ['up', 'down'],
        };
    }
    async getErrorsList(request, dateRangeStart, dateRangeEnd, filters) {
        const statusDown = {
            term: {
                'monitor.status': {
                    value: 'down',
                },
            },
        };
        const query = getFilteredQuery(dateRangeStart, dateRangeEnd, filters);
        if (lodash_1.get(query, 'bool.must', undefined)) {
            query.bool.must.push(statusDown);
        }
        else {
            lodash_1.set(query, 'bool.must', [{ ...statusDown }]);
        }
        const params = {
            index: constants_1.INDEX_NAMES.HEARTBEAT,
            body: {
                query,
                aggs: {
                    error_type: {
                        terms: {
                            field: 'error.type',
                        },
                        aggs: {
                            by_id: {
                                terms: {
                                    field: 'monitor.id',
                                },
                                aggs: {
                                    latest: {
                                        top_hits: {
                                            sort: [{ '@timestamp': { order: 'desc' } }],
                                            size: 1,
                                        },
                                    },
                                },
                            },
                        },
                    },
                },
            },
        };
        const queryResult = await this.database.search(request, params);
        const errorsList = [];
        lodash_1.get(queryResult, 'aggregations.error_type.buckets', []).forEach(({ key: errorType, by_id: { buckets: monitorBuckets }, }) => {
            monitorBuckets.forEach(bucket => {
                const count = lodash_1.get(bucket, 'doc_count', null);
                const monitorId = lodash_1.get(bucket, 'key', null);
                const source = lodash_1.get(bucket, 'latest.hits.hits[0]._source', null);
                const errorMessage = lodash_1.get(source, 'error.message', null);
                const statusCode = lodash_1.get(source, 'http.response.status_code', null);
                const timestamp = lodash_1.get(source, '@timestamp', null);
                const monitorType = lodash_1.get(source, 'monitor.type', null);
                errorsList.push({
                    latestMessage: errorMessage,
                    monitorId,
                    type: errorType,
                    monitorType,
                    count,
                    statusCode,
                    timestamp,
                });
            });
        });
        return errorsList;
    }
}
exports.ElasticsearchMonitorsAdapter = ElasticsearchMonitorsAdapter;
