"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const react_dom_1 = tslib_1.__importDefault(require("react-dom"));
const react_dom_2 = require("react-dom");
const chrome_1 = tslib_1.__importDefault(require("ui/chrome"));
const constants_1 = require("../../../../common/constants");
class UMKibanaFrameworkAdapter {
    constructor(uiRoutes, dateRangeStart, dateRangeEnd, autorefreshInterval, autorefreshIsPaused) {
        this.render = (renderComponent, createGraphQLClient) => {
            const route = {
                controllerAs: 'uptime',
                // @ts-ignore angular
                controller: ($scope, $route, $http, config) => {
                    const graphQLClient = createGraphQLClient(this.uriPath, this.xsrfHeader);
                    config.bindToScope($scope, 'k7design');
                    $scope.$$postDigest(() => {
                        const elem = document.getElementById('uptimeReactRoot');
                        let kibanaBreadcrumbs = [];
                        if ($scope.k7design) {
                            chrome_1.default.breadcrumbs.get$().subscribe((breadcrumbs) => {
                                kibanaBreadcrumbs = breadcrumbs;
                            });
                        }
                        const basePath = chrome_1.default.getBasePath();
                        const routerBasename = basePath.endsWith('/')
                            ? `${basePath}/${constants_1.PLUGIN.ROUTER_BASE_NAME}`
                            : basePath + constants_1.PLUGIN.ROUTER_BASE_NAME;
                        const persistedState = this.initializePersistedState();
                        const darkMode = config.get('theme:darkMode', false) || false;
                        const { autorefreshIsPaused, autorefreshInterval, dateRangeStart, dateRangeEnd, } = persistedState;
                        react_dom_1.default.render(renderComponent({
                            basePath,
                            darkMode,
                            isUsingK7Design: $scope.k7design,
                            updateBreadcrumbs: chrome_1.default.breadcrumbs.set,
                            kibanaBreadcrumbs,
                            routerBasename,
                            graphQLClient,
                            initialAutorefreshIsPaused: autorefreshIsPaused,
                            initialAutorefreshInterval: autorefreshInterval,
                            initialDateRangeStart: dateRangeStart,
                            initialDateRangeEnd: dateRangeEnd,
                            persistState: this.updatePersistedState,
                        }), elem);
                        this.manageAngularLifecycle($scope, $route, elem);
                    });
                },
                template: '<uptime-app section="kibana" id="uptimeReactRoot" class="app-wrapper-panel"></uptime-app>',
            };
            this.uiRoutes.enable();
            // TODO: hack to refer all routes to same endpoint, use a more proper way of achieving this
            this.uiRoutes.otherwise(route);
        };
        // @ts-ignore angular params
        this.manageAngularLifecycle = ($scope, $route, elem) => {
            const lastRoute = $route.current;
            const deregister = $scope.$on('$locationChangeSuccess', () => {
                const currentRoute = $route.current;
                if (lastRoute.$$route && lastRoute.$$route.template === currentRoute.$$route.template) {
                    $route.current = lastRoute;
                }
            });
            $scope.$on('$destroy', () => {
                deregister();
                react_dom_2.unmountComponentAtNode(elem);
            });
        };
        this.initializePersistedState = () => {
            const uptimeConfigurationData = window.localStorage.getItem(constants_1.PLUGIN.LOCAL_STORAGE_KEY);
            const defaultState = {
                autorefreshIsPaused: this.defaultAutorefreshIsPaused,
                autorefreshInterval: this.defaultAutorefreshInterval,
                dateRangeStart: this.defaultDateRangeStart,
                dateRangeEnd: this.defaultDateRangeEnd,
            };
            try {
                if (uptimeConfigurationData) {
                    const parsed = JSON.parse(uptimeConfigurationData) || {};
                    const { dateRangeStart, dateRangeEnd } = parsed;
                    // TODO: this is defensive code to ensure we don't encounter problems
                    // when encountering older versions of the localStorage values.
                    // The old code has never been released, so users don't need it, and this
                    // code should be removed eventually.
                    if ((dateRangeEnd && typeof dateRangeEnd === 'number') ||
                        (dateRangeStart && typeof dateRangeStart === 'number')) {
                        this.updatePersistedState(defaultState);
                        return defaultState;
                    }
                    return parsed;
                }
            }
            catch (e) {
                // TODO: this should result in a redirect to error page
                throw e;
            }
            this.updatePersistedState(defaultState);
            return defaultState;
        };
        this.updatePersistedState = (state) => {
            window.localStorage.setItem(constants_1.PLUGIN.LOCAL_STORAGE_KEY, JSON.stringify(state));
        };
        this.uiRoutes = uiRoutes;
        this.xsrfHeader = chrome_1.default.getXsrfToken();
        this.uriPath = `${chrome_1.default.getBasePath()}/api/uptime/graphql`;
        this.defaultDateRangeStart = dateRangeStart || 'now-15m';
        this.defaultDateRangeEnd = dateRangeEnd || 'now';
        this.defaultAutorefreshInterval = autorefreshInterval || 60 * 1000;
        this.defaultAutorefreshIsPaused = autorefreshIsPaused || true;
    }
}
exports.UMKibanaFrameworkAdapter = UMKibanaFrameworkAdapter;
