"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const boom_1 = tslib_1.__importDefault(require("boom"));
const types_1 = require("../../common/types");
const es_version_precheck_1 = require("../lib/es_version_precheck");
const reindexing_1 = require("../lib/reindexing");
const reindex_actions_1 = require("../lib/reindexing/reindex_actions");
function registerReindexWorker(server, credentialStore) {
    const { callWithRequest, callWithInternalUser } = server.plugins.elasticsearch.getCluster('admin');
    const xpackInfo = server.plugins.xpack_main.info;
    const savedObjectsRepository = server.savedObjects.getSavedObjectsRepository(callWithInternalUser);
    const savedObjectsClient = new server.savedObjects.SavedObjectsClient(savedObjectsRepository);
    // Cannot pass server.log directly because it's value changes during startup (?).
    // Use this function to proxy through.
    const log = (tags, data, timestamp) => server.log(tags, data, timestamp);
    const worker = new reindexing_1.ReindexWorker(savedObjectsClient, credentialStore, callWithRequest, callWithInternalUser, xpackInfo, log);
    // Wait for ES connection before starting the polling loop.
    server.plugins.elasticsearch.waitUntilReady().then(() => {
        worker.start();
        server.events.on('stop', () => worker.stop());
    });
    return worker;
}
exports.registerReindexWorker = registerReindexWorker;
function registerReindexIndicesRoutes(server, worker, credentialStore) {
    const { callWithRequest } = server.plugins.elasticsearch.getCluster('admin');
    const xpackInfo = server.plugins.xpack_main.info;
    const BASE_PATH = '/api/upgrade_assistant/reindex';
    // Start reindex for an index
    server.route({
        path: `${BASE_PATH}/{indexName}`,
        method: 'POST',
        options: {
            pre: [es_version_precheck_1.EsVersionPrecheck],
        },
        async handler(request) {
            const client = request.getSavedObjectsClient();
            const { indexName } = request.params;
            const callCluster = callWithRequest.bind(null, request);
            const reindexActions = reindex_actions_1.reindexActionsFactory(client, callCluster);
            const reindexService = reindexing_1.reindexServiceFactory(callCluster, xpackInfo, reindexActions, server.log);
            try {
                if (!(await reindexService.hasRequiredPrivileges(indexName))) {
                    throw boom_1.default.forbidden(`You do not have adequate privileges to reindex this index.`);
                }
                const existingOp = await reindexService.findReindexOperation(indexName);
                // If the reindexOp already exists and it's paused, resume it. Otherwise create a new one.
                const reindexOp = existingOp && existingOp.attributes.status === types_1.ReindexStatus.paused
                    ? await reindexService.resumeReindexOperation(indexName)
                    : await reindexService.createReindexOperation(indexName);
                // Add users credentials for the worker to use
                credentialStore.set(reindexOp, request.headers);
                // Kick the worker on this node to immediately pickup the new reindex operation.
                worker.forceRefresh();
                return reindexOp.attributes;
            }
            catch (e) {
                if (!e.isBoom) {
                    return boom_1.default.boomify(e, { statusCode: 500 });
                }
                return e;
            }
        },
    });
    // Get status
    server.route({
        path: `${BASE_PATH}/{indexName}`,
        method: 'GET',
        options: {
            pre: [es_version_precheck_1.EsVersionPrecheck],
        },
        async handler(request) {
            const client = request.getSavedObjectsClient();
            const { indexName } = request.params;
            const callCluster = callWithRequest.bind(null, request);
            const reindexActions = reindex_actions_1.reindexActionsFactory(client, callCluster);
            const reindexService = reindexing_1.reindexServiceFactory(callCluster, xpackInfo, reindexActions, server.log);
            try {
                const hasRequiredPrivileges = await reindexService.hasRequiredPrivileges(indexName);
                const reindexOp = await reindexService.findReindexOperation(indexName);
                // If the user doesn't have privileges than querying for warnings is going to fail.
                const warnings = hasRequiredPrivileges
                    ? await reindexService.detectReindexWarnings(indexName)
                    : [];
                const indexGroup = reindexService.getIndexGroup(indexName);
                return {
                    reindexOp: reindexOp ? reindexOp.attributes : null,
                    warnings,
                    indexGroup,
                    hasRequiredPrivileges,
                };
            }
            catch (e) {
                if (!e.isBoom) {
                    return boom_1.default.boomify(e, { statusCode: 500 });
                }
                return e;
            }
        },
    });
    // Cancel reindex
    server.route({
        path: `${BASE_PATH}/{indexName}/cancel`,
        method: 'POST',
        options: {
            pre: [es_version_precheck_1.EsVersionPrecheck],
        },
        async handler(request) {
            const client = request.getSavedObjectsClient();
            const { indexName } = request.params;
            const callCluster = callWithRequest.bind(null, request);
            const reindexActions = reindex_actions_1.reindexActionsFactory(client, callCluster);
            const reindexService = reindexing_1.reindexServiceFactory(callCluster, xpackInfo, reindexActions, server.log);
            try {
                await reindexService.cancelReindexing(indexName);
                return { acknowledged: true };
            }
            catch (e) {
                if (!e.isBoom) {
                    return boom_1.default.boomify(e, { statusCode: 500 });
                }
                return e;
            }
        },
    });
}
exports.registerReindexIndicesRoutes = registerReindexIndicesRoutes;
