"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const boom_1 = tslib_1.__importDefault(require("boom"));
const types_1 = require("../../../common/types");
const index_settings_1 = require("./index_settings");
const VERSION_REGEX = new RegExp(/^([1-9]+)\.([0-9]+)\.([0-9]+)/);
const ML_INDICES = ['.ml-state', '.ml-anomalies', '.ml-config'];
const WATCHER_INDICES = ['.watches', '.triggered-watches'];
exports.reindexServiceFactory = (callCluster, xpackInfo, actions, log) => {
    // ------ Utility functions
    /**
     * If the index is a ML index that will cause jobs to fail when set to readonly,
     * turn on 'upgrade mode' to pause all ML jobs.
     * @param reindexOp
     */
    const stopMlJobs = async () => {
        await actions.incrementIndexGroupReindexes(types_1.IndexGroup.ml);
        await actions.runWhileIndexGroupLocked(types_1.IndexGroup.ml, async (mlDoc) => {
            await validateNodesMinimumVersion(6, 7);
            const res = await callCluster('transport.request', {
                path: '/_xpack/ml/set_upgrade_mode?enabled=true',
                method: 'POST',
            });
            if (!res.acknowledged) {
                throw new Error(`Could not stop ML jobs`);
            }
            return mlDoc;
        });
    };
    /**
     * Resumes ML jobs if there are no more remaining reindex operations.
     */
    const resumeMlJobs = async () => {
        await actions.decrementIndexGroupReindexes(types_1.IndexGroup.ml);
        await actions.runWhileIndexGroupLocked(types_1.IndexGroup.ml, async (mlDoc) => {
            if (mlDoc.attributes.runningReindexCount === 0) {
                const res = await callCluster('transport.request', {
                    path: '/_xpack/ml/set_upgrade_mode?enabled=false',
                    method: 'POST',
                });
                if (!res.acknowledged) {
                    throw new Error(`Could not resume ML jobs`);
                }
            }
            return mlDoc;
        });
    };
    /**
     * Stops Watcher in Elasticsearch.
     */
    const stopWatcher = async () => {
        await actions.incrementIndexGroupReindexes(types_1.IndexGroup.watcher);
        await actions.runWhileIndexGroupLocked(types_1.IndexGroup.watcher, async (watcherDoc) => {
            const { acknowledged } = await callCluster('transport.request', {
                path: '/_xpack/watcher/_stop',
                method: 'POST',
            });
            if (!acknowledged) {
                throw new Error('Could not stop Watcher');
            }
            return watcherDoc;
        });
    };
    /**
     * Starts Watcher in Elasticsearch.
     */
    const startWatcher = async () => {
        await actions.decrementIndexGroupReindexes(types_1.IndexGroup.watcher);
        await actions.runWhileIndexGroupLocked(types_1.IndexGroup.watcher, async (watcherDoc) => {
            if (watcherDoc.attributes.runningReindexCount === 0) {
                const { acknowledged } = await callCluster('transport.request', {
                    path: '/_xpack/watcher/_start',
                    method: 'POST',
                });
                if (!acknowledged) {
                    throw new Error('Could not start Watcher');
                }
            }
            return watcherDoc;
        });
    };
    const cleanupChanges = async (reindexOp) => {
        // Cancel reindex task if it was started but not completed
        if (reindexOp.attributes.lastCompletedStep === types_1.ReindexStep.reindexStarted) {
            await callCluster('tasks.cancel', {
                taskId: reindexOp.attributes.reindexTaskId,
            }).catch(e => undefined); // Ignore any exceptions trying to cancel (it may have already completed).
        }
        // Set index back to writable if we ever got past this point.
        if (reindexOp.attributes.lastCompletedStep >= types_1.ReindexStep.readonly) {
            await callCluster('indices.putSettings', {
                index: reindexOp.attributes.indexName,
                body: { 'index.blocks.write': false },
            });
        }
        if (reindexOp.attributes.lastCompletedStep >= types_1.ReindexStep.newIndexCreated &&
            reindexOp.attributes.lastCompletedStep < types_1.ReindexStep.aliasCreated) {
            await callCluster('indices.delete', { index: reindexOp.attributes.newIndexName });
        }
        // Resume consumers if we ever got past this point.
        if (reindexOp.attributes.lastCompletedStep >= types_1.ReindexStep.indexGroupServicesStopped) {
            await resumeIndexGroupServices(reindexOp);
        }
        return reindexOp;
    };
    // ------ Functions used to process the state machine
    const validateNodesMinimumVersion = async (minMajor, minMinor) => {
        const nodesResponse = await callCluster('transport.request', {
            path: '/_nodes',
            method: 'GET',
        });
        const outDatedNodes = Object.values(nodesResponse.nodes).filter((node) => {
            const matches = node.version.match(VERSION_REGEX);
            const major = parseInt(matches[1], 10);
            const minor = parseInt(matches[2], 10);
            // All ES nodes must be >= 6.7.0 to pause ML jobs
            return !(major > minMajor || (major === minMajor && minor >= minMinor));
        });
        if (outDatedNodes.length > 0) {
            const nodeList = JSON.stringify(outDatedNodes.map((n) => n.name));
            throw new Error(`Some nodes are not on minimum version (${minMajor}.${minMinor}.0)  required: ${nodeList}`);
        }
    };
    const stopIndexGroupServices = async (reindexOp) => {
        if (exports.isMlIndex(reindexOp.attributes.indexName)) {
            await stopMlJobs();
        }
        else if (exports.isWatcherIndex(reindexOp.attributes.indexName)) {
            await stopWatcher();
        }
        return actions.updateReindexOp(reindexOp, {
            lastCompletedStep: types_1.ReindexStep.indexGroupServicesStopped,
        });
    };
    /**
     * Sets the original index as readonly so new data can be indexed until the reindex
     * is completed.
     * @param reindexOp
     */
    const setReadonly = async (reindexOp) => {
        const { indexName } = reindexOp.attributes;
        const putReadonly = await callCluster('indices.putSettings', {
            index: indexName,
            body: { 'index.blocks.write': true },
        });
        if (!putReadonly.acknowledged) {
            throw new Error(`Index could not be set to readonly.`);
        }
        return actions.updateReindexOp(reindexOp, { lastCompletedStep: types_1.ReindexStep.readonly });
    };
    /**
     * Creates a new index with the same mappings and settings as the original index.
     * @param reindexOp
     */
    const createNewIndex = async (reindexOp) => {
        const { indexName, newIndexName } = reindexOp.attributes;
        const flatSettings = await actions.getFlatSettings(indexName);
        if (!flatSettings) {
            throw boom_1.default.notFound(`Index ${indexName} does not exist.`);
        }
        const { settings, mappings } = index_settings_1.transformFlatSettings(flatSettings);
        const createIndex = await callCluster('indices.create', {
            index: newIndexName,
            body: {
                settings,
                mappings,
            },
        });
        if (!createIndex.acknowledged) {
            throw boom_1.default.badImplementation(`Index could not be created: ${newIndexName}`);
        }
        return actions.updateReindexOp(reindexOp, {
            lastCompletedStep: types_1.ReindexStep.newIndexCreated,
        });
    };
    /**
     * Begins the reindex process via Elasticsearch's Reindex API.
     * @param reindexOp
     */
    const startReindexing = async (reindexOp) => {
        const { indexName } = reindexOp.attributes;
        const reindexBody = {
            source: { index: indexName },
            dest: { index: reindexOp.attributes.newIndexName },
        };
        const booleanFieldPaths = await actions.getBooleanFieldPaths(indexName);
        if (booleanFieldPaths.length) {
            reindexBody.script = {
                lang: 'painless',
                source: `
          // Updates a single field in a Map
          void updateField(Map data, String fieldName) {
            if (
              data[fieldName] == 'yes' ||
              data[fieldName] == '1' ||
              (data[fieldName] instanceof Integer && data[fieldName] == 1) ||
              data[fieldName] == 'on'
            ) {
              data[fieldName] = true;
            } else if (
              data[fieldName] == 'no' ||
              data[fieldName] == '0' ||
              (data[fieldName] instanceof Integer && data[fieldName] == 0) ||
              data[fieldName] == 'off'
            ) {
              data[fieldName] = false;
            }
          }

          // Recursively walks the fieldPath list and calls
          void updateFieldPath(def data, List fieldPath) {
            String pathHead = fieldPath[0];

            if (fieldPath.getLength() == 1) {
              if (data.get(pathHead) !== null) {
                updateField(data, pathHead);
              }
            } else {
              List fieldPathTail = fieldPath.subList(1, fieldPath.getLength());

              if (data.get(pathHead) instanceof List) {
                for (item in data[pathHead]) {
                  updateFieldPath(item, fieldPathTail);
                }
              } else if (data.get(pathHead) instanceof Map) {
                updateFieldPath(data[pathHead], fieldPathTail);
              }
            }
          }

          for (fieldPath in params.booleanFieldPaths) {
            updateFieldPath(ctx._source, fieldPath)
          }
        `,
                params: { booleanFieldPaths },
            };
        }
        const startReindex = (await callCluster('reindex', {
            refresh: true,
            waitForCompletion: false,
            body: reindexBody,
        }));
        return actions.updateReindexOp(reindexOp, {
            lastCompletedStep: types_1.ReindexStep.reindexStarted,
            reindexTaskId: startReindex.task,
            reindexTaskPercComplete: 0,
        });
    };
    /**
     * Polls Elasticsearch's Tasks API to see if the reindex operation has been completed.
     * @param reindexOp
     */
    const updateReindexStatus = async (reindexOp) => {
        const taskId = reindexOp.attributes.reindexTaskId;
        // Check reindexing task progress
        const taskResponse = await callCluster('tasks.get', {
            taskId,
            waitForCompletion: false,
        });
        if (!taskResponse.completed) {
            // Updated the percent complete
            const perc = taskResponse.task.status.created / taskResponse.task.status.total;
            return actions.updateReindexOp(reindexOp, {
                reindexTaskPercComplete: perc,
            });
        }
        else if (taskResponse.task.status.canceled === 'by user request') {
            // Set the status to cancelled
            reindexOp = await actions.updateReindexOp(reindexOp, {
                status: types_1.ReindexStatus.cancelled,
            });
            // Do any other cleanup work necessary
            reindexOp = await cleanupChanges(reindexOp);
        }
        else {
            // Check that it reindexed all documents
            const { count } = await callCluster('count', { index: reindexOp.attributes.indexName });
            if (taskResponse.task.status.created < count) {
                // Include the entire task result in the error message. This should be guaranteed
                // to be JSON-serializable since it just came back from Elasticsearch.
                throw boom_1.default.badData(`Reindexing failed: ${JSON.stringify(taskResponse)}`);
            }
            // Update the status
            reindexOp = await actions.updateReindexOp(reindexOp, {
                lastCompletedStep: types_1.ReindexStep.reindexCompleted,
                reindexTaskPercComplete: 1,
            });
        }
        // Delete the task from ES .tasks index
        const deleteTaskResp = await callCluster('delete', {
            index: '.tasks',
            type: 'task',
            id: taskId,
        });
        if (deleteTaskResp.result !== 'deleted') {
            throw boom_1.default.badImplementation(`Could not delete reindexing task ${taskId}`);
        }
        return reindexOp;
    };
    /**
     * Creates an alias that points the old index to the new index, deletes the old index.
     * @param reindexOp
     */
    const switchAlias = async (reindexOp) => {
        const { indexName, newIndexName } = reindexOp.attributes;
        const existingAliases = (await callCluster('indices.getAlias', {
            index: indexName,
        }))[indexName].aliases;
        const extraAlises = Object.keys(existingAliases).map(aliasName => ({
            add: { index: newIndexName, alias: aliasName, ...existingAliases[aliasName] },
        }));
        const aliasResponse = await callCluster('indices.updateAliases', {
            body: {
                actions: [
                    { add: { index: newIndexName, alias: indexName } },
                    { remove_index: { index: indexName } },
                    ...extraAlises,
                ],
            },
        });
        if (!aliasResponse.acknowledged) {
            throw boom_1.default.badImplementation(`Index aliases could not be created.`);
        }
        return actions.updateReindexOp(reindexOp, {
            lastCompletedStep: types_1.ReindexStep.aliasCreated,
        });
    };
    const resumeIndexGroupServices = async (reindexOp) => {
        if (exports.isMlIndex(reindexOp.attributes.indexName)) {
            await resumeMlJobs();
        }
        else if (exports.isWatcherIndex(reindexOp.attributes.indexName)) {
            await startWatcher();
        }
        // Only change the status if we're still in-progress (this function is also called when the reindex fails or is cancelled)
        if (reindexOp.attributes.status === types_1.ReindexStatus.inProgress) {
            return actions.updateReindexOp(reindexOp, {
                lastCompletedStep: types_1.ReindexStep.indexGroupServicesStarted,
            });
        }
        else {
            return reindexOp;
        }
    };
    // ------ The service itself
    return {
        async hasRequiredPrivileges(indexName) {
            // If security is disabled or unavailable, return true.
            const security = xpackInfo.feature('security');
            if (!security.isAvailable() || !security.isEnabled()) {
                return true;
            }
            const names = [indexName, index_settings_1.generateNewIndexName(indexName)];
            const sourceName = index_settings_1.sourceNameForIndex(indexName);
            // if we have re-indexed this in the past, there will be an
            // underlying alias we will also need to update.
            if (sourceName !== indexName) {
                names.push(sourceName);
            }
            // Otherwise, query for required privileges for this index.
            const body = {
                cluster: ['manage'],
                index: [
                    {
                        names,
                        allow_restricted_indices: true,
                        privileges: ['all'],
                    },
                    {
                        names: ['.tasks'],
                        privileges: ['read', 'delete'],
                    },
                ],
            };
            if (exports.isMlIndex(indexName)) {
                body.cluster = [...body.cluster, 'manage_ml'];
            }
            if (exports.isWatcherIndex(indexName)) {
                body.cluster = [...body.cluster, 'manage_watcher'];
            }
            const resp = await callCluster('transport.request', {
                path: '/_security/user/_has_privileges',
                method: 'POST',
                body,
            });
            return resp.has_all_requested;
        },
        async detectReindexWarnings(indexName) {
            const flatSettings = await actions.getFlatSettings(indexName);
            if (!flatSettings) {
                return null;
            }
            else {
                return index_settings_1.getReindexWarnings(flatSettings);
            }
        },
        getIndexGroup(indexName) {
            if (exports.isMlIndex(indexName)) {
                return types_1.IndexGroup.ml;
            }
            else if (exports.isWatcherIndex(indexName)) {
                return types_1.IndexGroup.watcher;
            }
        },
        async createReindexOperation(indexName) {
            const indexExists = await callCluster('indices.exists', { index: indexName });
            if (!indexExists) {
                throw boom_1.default.notFound(`Index ${indexName} does not exist in this cluster.`);
            }
            const existingReindexOps = await actions.findReindexOperations(indexName);
            if (existingReindexOps.total !== 0) {
                const existingOp = existingReindexOps.saved_objects[0];
                if (existingOp.attributes.status === types_1.ReindexStatus.failed ||
                    existingOp.attributes.status === types_1.ReindexStatus.cancelled) {
                    // Delete the existing one if it failed or was cancelled to give a chance to retry.
                    await actions.deleteReindexOp(existingOp);
                }
                else {
                    throw boom_1.default.badImplementation(`A reindex operation already in-progress for ${indexName}`);
                }
            }
            return actions.createReindexOp(indexName);
        },
        async findReindexOperation(indexName) {
            const findResponse = await actions.findReindexOperations(indexName);
            // Bail early if it does not exist or there is more than one.
            if (findResponse.total === 0) {
                return null;
            }
            else if (findResponse.total > 1) {
                throw boom_1.default.badImplementation(`More than one reindex operation found for ${indexName}`);
            }
            return findResponse.saved_objects[0];
        },
        findAllByStatus: actions.findAllByStatus,
        async processNextStep(reindexOp) {
            return actions.runWhileLocked(reindexOp, async (lockedReindexOp) => {
                try {
                    switch (lockedReindexOp.attributes.lastCompletedStep) {
                        case types_1.ReindexStep.created:
                            lockedReindexOp = await stopIndexGroupServices(lockedReindexOp);
                            break;
                        case types_1.ReindexStep.indexGroupServicesStopped:
                            lockedReindexOp = await setReadonly(lockedReindexOp);
                            break;
                        case types_1.ReindexStep.readonly:
                            lockedReindexOp = await createNewIndex(lockedReindexOp);
                            break;
                        case types_1.ReindexStep.newIndexCreated:
                            lockedReindexOp = await startReindexing(lockedReindexOp);
                            break;
                        case types_1.ReindexStep.reindexStarted:
                            lockedReindexOp = await updateReindexStatus(lockedReindexOp);
                            break;
                        case types_1.ReindexStep.reindexCompleted:
                            lockedReindexOp = await switchAlias(lockedReindexOp);
                            break;
                        case types_1.ReindexStep.aliasCreated:
                            lockedReindexOp = await resumeIndexGroupServices(lockedReindexOp);
                            break;
                        case types_1.ReindexStep.indexGroupServicesStarted:
                            lockedReindexOp = await actions.updateReindexOp(lockedReindexOp, {
                                status: types_1.ReindexStatus.completed,
                            });
                        default:
                            break;
                    }
                }
                catch (e) {
                    log(['upgrade_assistant', 'error'], `Reindexing step failed: ${e instanceof Error ? e.stack : e.toString()}`);
                    // Trap the exception and add the message to the object so the UI can display it.
                    lockedReindexOp = await actions.updateReindexOp(lockedReindexOp, {
                        status: types_1.ReindexStatus.failed,
                        errorMessage: e.toString(),
                    });
                    // Cleanup any changes, ignoring any errors.
                    lockedReindexOp = await cleanupChanges(lockedReindexOp).catch(e => lockedReindexOp);
                }
                return lockedReindexOp;
            });
        },
        async pauseReindexOperation(indexName) {
            const reindexOp = await this.findReindexOperation(indexName);
            if (!reindexOp) {
                throw new Error(`No reindex operation found for index ${indexName}`);
            }
            return actions.runWhileLocked(reindexOp, async (op) => {
                if (op.attributes.status === types_1.ReindexStatus.paused) {
                    // Another node already paused the operation, don't do anything
                    return reindexOp;
                }
                else if (op.attributes.status !== types_1.ReindexStatus.inProgress) {
                    throw new Error(`Reindex operation must be inProgress in order to be paused.`);
                }
                return actions.updateReindexOp(op, { status: types_1.ReindexStatus.paused });
            });
        },
        async resumeReindexOperation(indexName) {
            const reindexOp = await this.findReindexOperation(indexName);
            if (!reindexOp) {
                throw new Error(`No reindex operation found for index ${indexName}`);
            }
            return actions.runWhileLocked(reindexOp, async (op) => {
                if (op.attributes.status === types_1.ReindexStatus.inProgress) {
                    // Another node already resumed the operation, don't do anything
                    return reindexOp;
                }
                else if (op.attributes.status !== types_1.ReindexStatus.paused) {
                    throw new Error(`Reindex operation must be paused in order to be resumed.`);
                }
                return actions.updateReindexOp(op, { status: types_1.ReindexStatus.inProgress });
            });
        },
        async cancelReindexing(indexName) {
            const reindexOp = await this.findReindexOperation(indexName);
            if (!reindexOp) {
                throw new Error(`No reindex operation found for index ${indexName}`);
            }
            else if (reindexOp.attributes.status !== types_1.ReindexStatus.inProgress) {
                throw new Error(`Reindex operation is not in progress`);
            }
            else if (reindexOp.attributes.lastCompletedStep !== types_1.ReindexStep.reindexStarted) {
                throw new Error(`Reindex operation is not current waiting for reindex task to complete`);
            }
            const resp = await callCluster('tasks.cancel', {
                taskId: reindexOp.attributes.reindexTaskId,
            });
            if (resp.node_failures && resp.node_failures.length > 0) {
                throw new Error(`Could not cancel reindex.`);
            }
            return reindexOp;
        },
    };
};
exports.isMlIndex = (indexName) => {
    const sourceName = index_settings_1.sourceNameForIndex(indexName);
    return ML_INDICES.indexOf(sourceName) >= 0;
};
exports.isWatcherIndex = (indexName) => {
    const sourceName = index_settings_1.sourceNameForIndex(indexName);
    return WATCHER_INDICES.indexOf(sourceName) >= 0;
};
