"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const moment_1 = tslib_1.__importDefault(require("moment"));
const types_1 = require("../../../common/types");
const index_settings_1 = require("./index_settings");
// TODO: base on elasticsearch.requestTimeout?
exports.LOCK_WINDOW = moment_1.default.duration(90, 'seconds');
exports.reindexActionsFactory = (client, callCluster) => {
    // ----- Internal functions
    const isLocked = (reindexOp) => {
        if (reindexOp.attributes.locked) {
            const now = moment_1.default();
            const lockedTime = moment_1.default(reindexOp.attributes.locked);
            // If the object has been locked for more than the LOCK_WINDOW, assume the process that locked it died.
            if (now.subtract(exports.LOCK_WINDOW) < lockedTime) {
                return true;
            }
        }
        return false;
    };
    const acquireLock = async (reindexOp) => {
        if (isLocked(reindexOp)) {
            throw new Error(`Another Kibana process is currently modifying this reindex operation.`);
        }
        return client.update(types_1.REINDEX_OP_TYPE, reindexOp.id, { ...reindexOp.attributes, locked: moment_1.default().format() }, { version: reindexOp.version });
    };
    const releaseLock = (reindexOp) => {
        return client.update(types_1.REINDEX_OP_TYPE, reindexOp.id, { ...reindexOp.attributes, locked: null }, { version: reindexOp.version });
    };
    // ----- Public interface
    return {
        async createReindexOp(indexName) {
            return client.create(types_1.REINDEX_OP_TYPE, {
                indexName,
                newIndexName: index_settings_1.generateNewIndexName(indexName),
                status: types_1.ReindexStatus.inProgress,
                lastCompletedStep: types_1.ReindexStep.created,
                locked: null,
                reindexTaskId: null,
                reindexTaskPercComplete: null,
                errorMessage: null,
                runningReindexCount: null,
            });
        },
        deleteReindexOp(reindexOp) {
            return client.delete(types_1.REINDEX_OP_TYPE, reindexOp.id);
        },
        async updateReindexOp(reindexOp, attrs = {}) {
            if (!isLocked(reindexOp)) {
                throw new Error(`ReindexOperation must be locked before updating.`);
            }
            const newAttrs = { ...reindexOp.attributes, locked: moment_1.default().format(), ...attrs };
            return client.update(types_1.REINDEX_OP_TYPE, reindexOp.id, newAttrs, {
                version: reindexOp.version,
            });
        },
        async runWhileLocked(reindexOp, func) {
            reindexOp = await acquireLock(reindexOp);
            try {
                reindexOp = await func(reindexOp);
            }
            finally {
                reindexOp = await releaseLock(reindexOp);
            }
            return reindexOp;
        },
        findReindexOperations(indexName) {
            return client.find({
                type: types_1.REINDEX_OP_TYPE,
                search: `"${indexName}"`,
                searchFields: ['indexName'],
            });
        },
        async findAllByStatus(status) {
            const firstPage = await client.find({
                type: types_1.REINDEX_OP_TYPE,
                search: status.toString(),
                searchFields: ['status'],
            });
            if (firstPage.total === firstPage.saved_objects.length) {
                return firstPage.saved_objects;
            }
            let allOps = firstPage.saved_objects;
            let page = firstPage.page + 1;
            while (allOps.length < firstPage.total) {
                const nextPage = await client.find({
                    type: types_1.REINDEX_OP_TYPE,
                    search: status.toString(),
                    searchFields: ['status'],
                    page,
                });
                allOps = [...allOps, ...nextPage.saved_objects];
                page++;
            }
            return allOps;
        },
        async getBooleanFieldPaths(indexName) {
            const results = await callCluster('indices.getMapping', {
                index: indexName,
                include_type_name: true,
            });
            const mapping = index_settings_1.getSingleMappingType(results[indexName].mappings);
            // It's possible an index doesn't have a mapping.
            return mapping && mapping.properties ? index_settings_1.findBooleanFields(mapping.properties) : [];
        },
        async getFlatSettings(indexName) {
            const flatSettings = (await callCluster('transport.request', {
                path: `/${encodeURIComponent(indexName)}?include_type_name=true&flat_settings=true`,
            }));
            if (!flatSettings[indexName]) {
                return null;
            }
            return flatSettings[indexName];
        },
        async _fetchAndLockIndexGroupDoc(indexGroup) {
            const fetchDoc = async () => {
                try {
                    // The IndexGroup enum value (a string) serves as the ID of the lock doc
                    return await client.get(types_1.REINDEX_OP_TYPE, indexGroup);
                }
                catch (e) {
                    if (e.isBoom && e.output.statusCode === 404) {
                        return await client.create(types_1.REINDEX_OP_TYPE, {
                            indexName: null,
                            newIndexName: null,
                            locked: null,
                            status: null,
                            lastCompletedStep: null,
                            reindexTaskId: null,
                            reindexTaskPercComplete: null,
                            errorMessage: null,
                            runningReindexCount: 0,
                        }, { id: indexGroup });
                    }
                    else {
                        throw e;
                    }
                }
            };
            const lockDoc = async (attempt = 1) => {
                try {
                    // Refetch the document each time to avoid version conflicts.
                    return await acquireLock(await fetchDoc());
                }
                catch (e) {
                    if (attempt >= 10) {
                        throw new Error(`Could not acquire lock for ML jobs`);
                    }
                    await new Promise(resolve => setTimeout(resolve, 1000));
                    return lockDoc(attempt + 1);
                }
            };
            return lockDoc();
        },
        async incrementIndexGroupReindexes(indexGroup) {
            this.runWhileIndexGroupLocked(indexGroup, lockDoc => this.updateReindexOp(lockDoc, {
                runningReindexCount: lockDoc.attributes.runningReindexCount + 1,
            }));
        },
        async decrementIndexGroupReindexes(indexGroup) {
            this.runWhileIndexGroupLocked(indexGroup, lockDoc => this.updateReindexOp(lockDoc, {
                runningReindexCount: lockDoc.attributes.runningReindexCount - 1,
            }));
        },
        async runWhileIndexGroupLocked(indexGroup, func) {
            let lockDoc = await this._fetchAndLockIndexGroupDoc(indexGroup);
            try {
                lockDoc = await func(lockDoc);
            }
            finally {
                await releaseLock(lockDoc);
            }
        },
    };
};
