"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const boom_1 = tslib_1.__importDefault(require("boom"));
const lodash_1 = require("lodash");
/**
 * Adds the index.query.default_field setting, generated from the index's mapping.
 *
 * @param callWithRequest
 * @param request
 * @param indexName
 * @param fieldTypes - Elasticsearch field types that should be used to generate the default_field from the index mapping
 * @param otherFields - Other fields that should be included in the generated default_field that do not match `fieldTypes`
 */
exports.addDefaultField = async (callWithRequest, request, indexName, fieldTypes, otherFields = new Set()) => {
    // Verify index.query.default_field is not already set.
    const settings = await callWithRequest(request, 'indices.getSettings', {
        index: indexName,
    });
    if (lodash_1.get(settings, `${indexName}.settings.index.query.default_field`)) {
        throw boom_1.default.badRequest(`Index ${indexName} already has index.query.default_field set`);
    }
    // Get the mapping and generate the default_field based on `fieldTypes`
    const mappingResp = await callWithRequest(request, 'indices.getMapping', {
        index: indexName,
        include_type_name: true,
    });
    const typeName = Object.getOwnPropertyNames(mappingResp[indexName].mappings)[0];
    const mapping = mappingResp[indexName].mappings[typeName].properties;
    const generatedDefaultFields = new Set(exports.generateDefaultFields(mapping, fieldTypes));
    // Update the setting with the generated default_field
    return await callWithRequest(request, 'indices.putSettings', {
        index: indexName,
        body: {
            index: { query: { default_field: [...generatedDefaultFields, ...otherFields] } },
        },
    });
};
/**
 * Recursively walks an index mapping and returns a flat array of dot-delimited
 * strings represent all fields that are of a type included in `DEFAULT_FIELD_TYPES`
 * @param mapping
 */
exports.generateDefaultFields = (mapping, fieldTypes) => Object.getOwnPropertyNames(mapping).reduce((defaultFields, fieldName) => {
    const { type, properties } = mapping[fieldName];
    if (type && fieldTypes.has(type)) {
        defaultFields.push(fieldName);
    }
    else if (properties) {
        exports.generateDefaultFields(properties, fieldTypes).forEach(subField => defaultFields.push(`${fieldName}.${subField}`));
    }
    return defaultFields;
}, []);
