"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const react_1 = tslib_1.__importDefault(require("react"));
const types_1 = require("../../../types");
const version_1 = require("x-pack/plugins/upgrade_assistant/common/version");
const constants_1 = require("../constants");
const cell_1 = require("./cell");
const index_table_1 = require("./index_table");
const OLD_INDEX_MESSAGE = `Index created before ${version_1.CURRENT_MAJOR_VERSION}.0`;
const DELETE_INDEX_MESSAGE = `.tasks index must be re-created`;
const NEEDS_DEFAULT_FIELD_MESSAGE = 'Number of fields exceeds automatic field expansion limit';
const sortByLevelDesc = (a, b) => {
    return -1 * (constants_1.LEVEL_MAP[a.level] - constants_1.LEVEL_MAP[b.level]);
};
/**
 * Used to show a single deprecation message with any detailed information.
 */
const MessageDeprecation = ({ deprecation, }) => {
    const items = [];
    if (deprecation.details) {
        items.push({ body: deprecation.details });
    }
    return (react_1.default.createElement(cell_1.DeprecationCell, { headline: deprecation.message, healthColor: constants_1.COLOR_MAP[deprecation.level], indexName: deprecation.index, reindex: deprecation.message === OLD_INDEX_MESSAGE, deleteIndex: deprecation.message === DELETE_INDEX_MESSAGE, needsDefaultFields: deprecation.message === NEEDS_DEFAULT_FIELD_MESSAGE, docUrl: deprecation.url, items: items }));
};
/**
 * Used to show a single (simple) deprecation message with any detailed information.
 */
const SimpleMessageDeprecation = ({ deprecation, }) => {
    const items = [];
    if (deprecation.details) {
        items.push({ body: deprecation.details });
    }
    return react_1.default.createElement(cell_1.DeprecationCell, { items: items, docUrl: deprecation.url });
};
/**
 * Shows a single deprecation and table of affected indices with details for each index.
 */
const IndexDeprecation = ({ deprecation, indices }) => {
    return (react_1.default.createElement(cell_1.DeprecationCell, { docUrl: deprecation.url },
        react_1.default.createElement(index_table_1.IndexDeprecationTable, { indices: indices })));
};
/**
 * A list of deprecations that is either shown as individual deprecation cells or as a
 * deprecation summary for a list of indices.
 */
exports.DeprecationList = ({ deprecations, currentGroupBy }) => {
    // If we're grouping by message and the first deprecation has an index field, show an index
    // group deprecation. Otherwise, show each message.
    if (currentGroupBy === types_1.GroupByOption.message && deprecations[0].index !== undefined) {
        // We assume that every deprecation message is the same issue (since they have the same
        // message) and that each deprecation will have an index associated with it.
        const indices = deprecations.map(dep => ({
            index: dep.index,
            details: dep.details,
            reindex: dep.message === OLD_INDEX_MESSAGE,
            delete: dep.message === DELETE_INDEX_MESSAGE,
            needsDefaultFields: dep.message === NEEDS_DEFAULT_FIELD_MESSAGE,
        }));
        return react_1.default.createElement(IndexDeprecation, { indices: indices, deprecation: deprecations[0] });
    }
    else if (currentGroupBy === types_1.GroupByOption.index) {
        return (react_1.default.createElement("div", null, deprecations.sort(sortByLevelDesc).map(dep => (react_1.default.createElement(MessageDeprecation, { deprecation: dep, key: dep.message })))));
    }
    else {
        return (react_1.default.createElement("div", null, deprecations.sort(sortByLevelDesc).map(dep => (react_1.default.createElement(SimpleMessageDeprecation, { deprecation: dep, key: dep.message })))));
    }
};
