"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const lodash_1 = require("lodash");
const react_1 = tslib_1.__importDefault(require("react"));
const eui_1 = require("@elastic/eui");
const react_2 = require("@kbn/i18n/react");
const button_1 = require("./default_fields/button");
const delete_tasks_button_1 = require("./delete_tasks_button");
const reindex_1 = require("./reindex");
const PAGE_SIZES = [10, 25, 50, 100, 250, 500, 1000];
class IndexDeprecationTableUI extends react_1.default.Component {
    constructor(props) {
        super(props);
        this.onTableChange = (tableProps) => {
            this.setState({
                sortField: tableProps.sort.field,
                sortDirection: tableProps.sort.direction,
                pageIndex: tableProps.page.index,
                pageSize: tableProps.page.size,
            });
        };
        this.state = {
            sortField: 'index',
            sortDirection: 'asc',
            pageIndex: 0,
            pageSize: 10,
        };
    }
    render() {
        const { intl } = this.props;
        const { pageIndex, pageSize, sortField, sortDirection } = this.state;
        const columns = [
            {
                field: 'index',
                name: intl.formatMessage({
                    id: 'xpack.upgradeAssistant.checkupTab.deprecations.indexTable.indexColumnLabel',
                    defaultMessage: 'Index',
                }),
                sortable: true,
            },
            {
                field: 'details',
                name: intl.formatMessage({
                    id: 'xpack.upgradeAssistant.checkupTab.deprecations.indexTable.detailsColumnLabel',
                    defaultMessage: 'Details',
                }),
            },
        ];
        if (this.actionsColumn) {
            // @ts-ignore
            columns.push(this.actionsColumn);
        }
        const sorting = { sort: { field: sortField, direction: sortDirection } };
        const pagination = {
            pageIndex,
            pageSize,
            ...this.pageSizeOptions(),
        };
        return (react_1.default.createElement(eui_1.EuiBasicTable, { items: this.getRows(), columns: columns, sorting: sorting, pagination: pagination, onChange: this.onTableChange, hasActions: false }));
    }
    getRows() {
        const { sortField, sortDirection, pageIndex, pageSize } = this.state;
        const { indices } = this.props;
        let sorted = lodash_1.sortBy(indices, sortField);
        if (sortDirection === 'desc') {
            sorted = sorted.reverse();
        }
        const start = pageIndex * pageSize;
        return sorted.slice(start, start + pageSize);
    }
    pageSizeOptions() {
        const { indices } = this.props;
        const totalItemCount = indices.length;
        // If we only have that smallest page size, don't show any page size options.
        if (totalItemCount <= PAGE_SIZES[0]) {
            return { totalItemCount, pageSizeOptions: [], hidePerPageOptions: true };
        }
        // Keep a size option if the # of items is larger than the previous option.
        // This avoids having a long list of useless page sizes.
        const pageSizeOptions = PAGE_SIZES.filter((perPage, idx) => {
            return idx === 0 || totalItemCount > PAGE_SIZES[idx - 1];
        });
        return { totalItemCount, pageSizeOptions, hidePerPageOptions: false };
    }
    get actionsColumn() {
        // NOTE: this naive implementation assumes all indices in the table are
        // should show the reindex button. This should work for known usecases.
        const { indices } = this.props;
        const showDeleteButton = indices.find(i => i.delete === true);
        const showReindexButton = indices.find(i => i.reindex === true);
        const showNeedsDefaultFieldsButton = indices.find(i => i.needsDefaultFields === true);
        if (!showDeleteButton && !showReindexButton && !showNeedsDefaultFieldsButton) {
            return null;
        }
        return {
            actions: [
                {
                    render(indexDep) {
                        if (showDeleteButton) {
                            return react_1.default.createElement(delete_tasks_button_1.DeleteTasksButton, null);
                        }
                        else if (showReindexButton) {
                            return react_1.default.createElement(reindex_1.ReindexButton, { indexName: indexDep.index });
                        }
                        else {
                            return react_1.default.createElement(button_1.FixDefaultFieldsButton, { indexName: indexDep.index });
                        }
                    },
                },
            ],
        };
    }
}
exports.IndexDeprecationTableUI = IndexDeprecationTableUI;
exports.IndexDeprecationTable = react_2.injectI18n(IndexDeprecationTableUI);
