"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const axios_1 = tslib_1.__importDefault(require("axios"));
const dedent_1 = tslib_1.__importDefault(require("dedent"));
const react_1 = tslib_1.__importStar(require("react"));
const eui_1 = require("@elastic/eui");
const chrome_1 = tslib_1.__importDefault(require("ui/chrome"));
const version_1 = require("x-pack/plugins/upgrade_assistant/common/version");
const types_1 = require("../../../types");
const XSRF = chrome_1.default.getXsrfToken();
exports.APIClient = axios_1.default.create({
    headers: {
        Accept: 'application/json',
        credentials: 'same-origin',
        'Content-Type': 'application/json',
        'kbn-version': XSRF,
        'kbn-xsrf': XSRF,
    },
});
const BACKUP_CONSOLE_LINK = chrome_1.default.addBasePath(`/app/kibana#/dev_tools/console?command=${encodeURIComponent(dedent_1.default(`
        # Reindexes the .tasks into a backup index. If this index is large,
        # you may need to use the \`wait_for_completion=false\` flag. Documentation:
        # https://www.elastic.co/guide/en/elasticsearch/reference/current/docs-reindex.html#_url_parameters_3
        POST /_reindex
        {
          "source": { "index": ".tasks" },
          "dest": { "index": ".tasks-v${version_1.CURRENT_MAJOR_VERSION}-backup" }
        }
      `))}`);
/**
 * This button should only be used to reindex the .tasks index which does not have an upgrade path and must be deleted.
 * This component was not made to be generic in order to avoid any bugs that would allow someone to delete other indices.
 */
class DeleteTasksButton extends react_1.default.Component {
    constructor(props) {
        super(props);
        this.showFlyout = () => {
            this.setState({ flyoutVisible: true });
        };
        this.closeFlyout = () => {
            this.setState({ flyoutVisible: false });
        };
        this.backupAndDelete = async () => {
            try {
                this.setState({ deleteStatus: types_1.LoadingState.Loading });
                await exports.APIClient.post(chrome_1.default.addBasePath('/api/upgrade_assistant/delete_tasks_index'));
                this.setState({ deleteStatus: types_1.LoadingState.Success });
            }
            catch (e) {
                this.setState({
                    deleteStatus: types_1.LoadingState.Error,
                    error: e,
                });
            }
        };
        this.flyoutButtonLabel = () => {
            switch (this.state.deleteStatus) {
                case types_1.LoadingState.Error:
                    return 'Try again';
                case types_1.LoadingState.Loading:
                    return 'Deleting…';
                case types_1.LoadingState.Success:
                    return 'Done!';
                default:
                    return 'Delete';
            }
        };
        this.state = {
            flyoutVisible: false,
        };
    }
    render() {
        const { flyoutVisible } = this.state;
        const { deleteStatus } = this.state;
        const buttonProps = { size: 's', onClick: this.showFlyout };
        let buttonContent = 'Delete';
        switch (deleteStatus) {
            case types_1.LoadingState.Loading:
                buttonContent = (react_1.default.createElement("span", null,
                    react_1.default.createElement(eui_1.EuiLoadingSpinner, { className: "upgReindexButton__spinner", size: "m" }),
                    " Deleting\u2026"));
                break;
            case types_1.LoadingState.Success:
                buttonProps.color = 'secondary';
                buttonProps.iconSide = 'left';
                buttonProps.iconType = 'check';
                buttonContent = 'Done';
                break;
            case types_1.LoadingState.Error:
                buttonProps.color = 'danger';
                buttonProps.iconSide = 'left';
                buttonProps.iconType = 'cross';
                buttonContent = 'Failed';
                break;
        }
        return (react_1.default.createElement(react_1.Fragment, null,
            react_1.default.createElement(eui_1.EuiButton, Object.assign({}, buttonProps), buttonContent),
            flyoutVisible && this.renderFlyout()));
    }
    renderFlyout() {
        const { deleteStatus } = this.state;
        return (react_1.default.createElement(eui_1.EuiPortal, null,
            react_1.default.createElement(eui_1.EuiFlyout, { onClose: this.closeFlyout, size: "m" },
                react_1.default.createElement(eui_1.EuiFlyoutHeader, { hasBorder: true },
                    react_1.default.createElement("h2", null, "Delete .tasks")),
                react_1.default.createElement(eui_1.EuiFlyoutBody, null,
                    react_1.default.createElement(eui_1.EuiCallOut, { title: "The .tasks index must be deleted", color: "danger", iconType: "alert" }),
                    react_1.default.createElement(eui_1.EuiSpacer, null),
                    react_1.default.createElement(eui_1.EuiText, { grow: false },
                        react_1.default.createElement("p", null,
                            "This index contains the results of long-running tasks in Elasticsearch. To upgrade, you will need to delete this index. If you still need to access this old information you should",
                            ' ',
                            react_1.default.createElement(eui_1.EuiLink, { href: BACKUP_CONSOLE_LINK, target: "_blank" }, "reindex into a backup index"),
                            ' ',
                            "before proceeding."))),
                react_1.default.createElement(eui_1.EuiFlyoutFooter, null,
                    react_1.default.createElement(eui_1.EuiFlexGroup, { justifyContent: "spaceBetween" },
                        react_1.default.createElement(eui_1.EuiFlexItem, { grow: false },
                            react_1.default.createElement(eui_1.EuiButtonEmpty, { iconType: "cross", onClick: this.closeFlyout, flush: "left" }, "Cancel")),
                        react_1.default.createElement(eui_1.EuiFlexItem, { grow: false },
                            react_1.default.createElement(eui_1.EuiButton, { fill: true, color: "danger", onClick: this.backupAndDelete, disabled: deleteStatus === types_1.LoadingState.Loading || deleteStatus === types_1.LoadingState.Success, isLoading: deleteStatus === types_1.LoadingState.Loading }, this.flyoutButtonLabel())))))));
    }
}
exports.DeleteTasksButton = DeleteTasksButton;
