"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const axios_1 = tslib_1.__importDefault(require("axios"));
const lodash_1 = require("lodash");
const react_1 = tslib_1.__importDefault(require("react"));
const eui_1 = require("@elastic/eui");
const react_2 = require("@kbn/i18n/react");
const chrome_1 = tslib_1.__importDefault(require("ui/chrome"));
const kfetch_1 = require("ui/kfetch");
const checkup_1 = require("./tabs/checkup");
const overview_1 = require("./tabs/overview");
const types_1 = require("./types");
var ClusterUpgradeState;
(function (ClusterUpgradeState) {
    ClusterUpgradeState[ClusterUpgradeState["needsUpgrade"] = 0] = "needsUpgrade";
    ClusterUpgradeState[ClusterUpgradeState["partiallyUpgraded"] = 1] = "partiallyUpgraded";
    ClusterUpgradeState[ClusterUpgradeState["upgraded"] = 2] = "upgraded";
})(ClusterUpgradeState || (ClusterUpgradeState = {}));
class UpgradeAssistantTabsUI extends react_1.default.Component {
    constructor(props) {
        super(props);
        this.onTabClick = (selectedTab) => {
            const selectedTabIndex = lodash_1.findIndex(this.tabs, { id: selectedTab.id });
            if (selectedTabIndex === -1) {
                throw new Error(`Clicked tab did not exist in tabs array`);
            }
            // Send telemetry info about the current selected tab
            // only in case the clicked tab id it's different from the
            // current selected tab id
            if (this.tabs[this.state.selectedTabIndex].id !== selectedTab.id) {
                this.sendTelemetryInfo(selectedTab.id);
            }
            this.setSelectedTabIndex(selectedTabIndex);
        };
        this.setSelectedTabIndex = (selectedTabIndex) => {
            this.setState({ selectedTabIndex });
        };
        this.loadData = async () => {
            try {
                this.setState({ loadingState: types_1.LoadingState.Loading });
                const resp = await axios_1.default.get(chrome_1.default.addBasePath('/api/upgrade_assistant/status'));
                this.setState({
                    loadingState: types_1.LoadingState.Success,
                    checkupData: resp.data,
                });
            }
            catch (e) {
                if (lodash_1.get(e, 'response.status') === 426) {
                    this.setState({
                        loadingState: types_1.LoadingState.Success,
                        clusterUpgradeState: lodash_1.get(e, 'response.data.attributes.allNodesUpgraded', false)
                            ? ClusterUpgradeState.upgraded
                            : ClusterUpgradeState.partiallyUpgraded,
                    });
                }
                else {
                    this.setState({ loadingState: types_1.LoadingState.Error, loadingError: e });
                }
            }
        };
        this.state = {
            loadingState: types_1.LoadingState.Loading,
            clusterUpgradeState: ClusterUpgradeState.needsUpgrade,
            selectedTabIndex: 0,
            telemetryState: types_1.TelemetryState.Complete,
        };
    }
    async componentDidMount() {
        await this.loadData();
        // Send telemetry info about the default selected tab
        this.sendTelemetryInfo(this.tabs[this.state.selectedTabIndex].id);
    }
    render() {
        const { selectedTabIndex, telemetryState, clusterUpgradeState } = this.state;
        const tabs = this.tabs;
        if (clusterUpgradeState === ClusterUpgradeState.partiallyUpgraded) {
            return (react_1.default.createElement(eui_1.EuiPageContent, null,
                react_1.default.createElement(eui_1.EuiPageContentBody, null,
                    react_1.default.createElement(eui_1.EuiEmptyPrompt, { iconType: "logoElasticsearch", title: react_1.default.createElement("h2", null,
                            react_1.default.createElement(react_2.FormattedMessage, { id: "xpack.upgradeAssistant.tabs.upgradingInterstitial.upgradingTitle", defaultMessage: "Your cluster is upgrading" })), body: react_1.default.createElement("p", null,
                            react_1.default.createElement(react_2.FormattedMessage, { id: "xpack.upgradeAssistant.tabs.upgradingInterstitial.upgradingDescription", defaultMessage: "One or more Elasticsearch nodes have a newer version of\n                      Elasticsearch than Kibana. Once all your nodes are upgraded, upgrade Kibana." })) }))));
        }
        else if (clusterUpgradeState === ClusterUpgradeState.upgraded) {
            return (react_1.default.createElement(eui_1.EuiPageContent, null,
                react_1.default.createElement(eui_1.EuiPageContentBody, null,
                    react_1.default.createElement(eui_1.EuiEmptyPrompt, { iconType: "logoElasticsearch", title: react_1.default.createElement("h2", null,
                            react_1.default.createElement(react_2.FormattedMessage, { id: "xpack.upgradeAssistant.tabs.upgradingInterstitial.upgradeCompleteTitle", defaultMessage: "Your cluster has been upgraded" })), body: react_1.default.createElement("p", null,
                            react_1.default.createElement(react_2.FormattedMessage, { id: "xpack.upgradeAssistant.tabs.upgradingInterstitial.upgradeCompleteDescription", defaultMessage: "All Elasticsearch nodes have been upgraded. You may now upgrade Kibana." })) }))));
        }
        return (react_1.default.createElement(eui_1.EuiTabbedContent, { "data-test-subj": telemetryState === types_1.TelemetryState.Running ? 'upgradeAssistantTelemetryRunning' : undefined, tabs: tabs, onTabClick: this.onTabClick, selectedTab: tabs[selectedTabIndex] }));
    }
    get tabs() {
        const { intl } = this.props;
        const { loadingError, loadingState, checkupData } = this.state;
        const commonProps = {
            loadingError,
            loadingState,
            refreshCheckupData: this.loadData,
            setSelectedTabIndex: this.setSelectedTabIndex,
        };
        return [
            {
                id: 'overview',
                name: intl.formatMessage({
                    id: 'xpack.upgradeAssistant.overviewTab.overviewTabTitle',
                    defaultMessage: 'Overview',
                }),
                content: react_1.default.createElement(overview_1.OverviewTab, Object.assign({ checkupData: checkupData }, commonProps)),
            },
            {
                id: 'cluster',
                name: intl.formatMessage({
                    id: 'xpack.upgradeAssistant.checkupTab.clusterTabLabel',
                    defaultMessage: 'Cluster',
                }),
                content: (react_1.default.createElement(checkup_1.CheckupTab, Object.assign({ key: "cluster", deprecations: checkupData ? checkupData.cluster : undefined, checkupLabel: intl.formatMessage({
                        id: 'xpack.upgradeAssistant.tabs.checkupTab.clusterLabel',
                        defaultMessage: 'cluster',
                    }) }, commonProps))),
            },
            {
                id: 'indices',
                name: intl.formatMessage({
                    id: 'xpack.upgradeAssistant.checkupTab.indicesTabLabel',
                    defaultMessage: 'Indices',
                }),
                content: (react_1.default.createElement(checkup_1.CheckupTab, Object.assign({ key: "indices", deprecations: checkupData ? checkupData.indices : undefined, checkupLabel: intl.formatMessage({
                        id: 'xpack.upgradeAssistant.checkupTab.indexLabel',
                        defaultMessage: 'index',
                    }), showBackupWarning: true }, commonProps))),
            },
        ];
    }
    async sendTelemetryInfo(tabName) {
        // In case we don't have any data yet, we wanna to ignore the
        // telemetry info update
        if (this.state.loadingState !== types_1.LoadingState.Success) {
            return;
        }
        this.setState({ telemetryState: types_1.TelemetryState.Running });
        await kfetch_1.kfetch({
            pathname: '/api/upgrade_assistant/telemetry/ui_open',
            method: 'PUT',
            body: JSON.stringify(lodash_1.set({}, tabName, true)),
        });
        this.setState({ telemetryState: types_1.TelemetryState.Complete });
    }
}
exports.UpgradeAssistantTabsUI = UpgradeAssistantTabsUI;
exports.UpgradeAssistantTabs = react_2.injectI18n(UpgradeAssistantTabsUI);
