"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const eui_1 = require("@elastic/eui");
const react_1 = require("@kbn/i18n/react");
const react_2 = tslib_1.__importStar(require("react"));
const components_1 = require("../../../../../../../../spaces/public/components");
const role_1 = require("../../../../../../lib/role");
const privilege_selector_1 = require("./privilege_selector");
class PrivilegeSpaceTableUI extends react_2.Component {
    constructor() {
        super(...arguments);
        this.state = {
            searchTerm: '',
        };
        this.getTableColumns = (role, availablePrivileges = []) => {
            const { intl } = this.props;
            const columns = [
                {
                    field: 'space',
                    name: '',
                    width: '50px',
                    sortable: true,
                    render: (space) => {
                        if ('deleted' in space) {
                            return null;
                        }
                        return react_2.default.createElement(components_1.SpaceAvatar, { space: space, size: "s" });
                    },
                },
                {
                    field: 'space',
                    name: intl.formatMessage({
                        id: 'xpack.security.management.editRoles.privilegeSpaceTable.spaceName',
                        defaultMessage: 'Space',
                    }),
                    width: this.props.readonly ? '75%' : '50%',
                    render: (space) => {
                        if ('deleted' in space) {
                            return (react_2.default.createElement(eui_1.EuiText, { color: 'subdued' },
                                react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.security.management.editRoles.privilegeSpaceTable.deletedSpaceDescription", defaultMessage: "{value} (deleted)", values: { value: space.id } })));
                        }
                        else {
                            return react_2.default.createElement(eui_1.EuiText, null, space.name);
                        }
                    },
                },
                {
                    field: 'privilege',
                    name: intl.formatMessage({
                        id: 'xpack.security.management.editRoles.privilegeSpaceTable.privilegeName',
                        defaultMessage: 'Privilege',
                    }),
                    width: this.props.readonly ? '25%' : undefined,
                    render: (privilege, record) => {
                        if (this.props.readonly || record.space.deleted) {
                            return privilege;
                        }
                        return (react_2.default.createElement(privilege_selector_1.PrivilegeSelector, { "data-test-subj": 'privilege-space-table-priv', availablePrivileges: availablePrivileges, value: privilege, disabled: role_1.isReservedRole(role) || this.props.readonly, onChange: this.onSpacePermissionChange(record), compressed: true }));
                    },
                },
            ];
            if (!this.props.readonly) {
                columns.push({
                    name: intl.formatMessage({
                        id: 'xpack.security.management.editRoles.privilegeSpaceTable.actionsName',
                        defaultMessage: 'Actions',
                    }),
                    actions: [
                        {
                            render: (record) => {
                                return (react_2.default.createElement(eui_1.EuiButtonIcon, { "aria-label": 'Remove custom privileges for this space', color: 'danger', onClick: () => this.onDeleteSpacePermissionsClick(record), iconType: 'trash' }));
                            },
                        },
                    ],
                });
            }
            return columns;
        };
        this.onSpacePermissionChange = (record) => (selectedPrivilege) => {
            const { id: spaceId } = record.space;
            const updatedPrivileges = {
                ...this.props.spacePrivileges,
            };
            updatedPrivileges[spaceId] = [selectedPrivilege];
            if (this.props.onChange) {
                this.props.onChange(updatedPrivileges);
            }
        };
        this.onDeleteSpacePermissionsClick = (record) => {
            const { id: spaceId } = record.space;
            const updatedPrivileges = {
                ...this.props.spacePrivileges,
            };
            delete updatedPrivileges[spaceId];
            if (this.props.onChange) {
                this.props.onChange(updatedPrivileges);
            }
        };
    }
    render() {
        const { role, spaces, availablePrivileges, spacePrivileges, intl } = this.props;
        const { searchTerm } = this.state;
        const allTableItems = Object.keys(spacePrivileges)
            .map(spaceId => {
            return {
                space: spaces.find(s => s.id === spaceId) || { id: spaceId, name: '', deleted: true },
                privilege: spacePrivileges[spaceId][0],
            };
        })
            .sort(item1 => {
            const isDeleted = 'deleted' in item1.space;
            return isDeleted ? 1 : -1;
        });
        const visibleTableItems = allTableItems.filter(item => {
            const isDeleted = 'deleted' in item.space;
            const searchField = isDeleted ? item.space.id : item.space.name;
            return searchField.toLowerCase().indexOf(searchTerm) >= 0;
        });
        if (allTableItems.length === 0) {
            return null;
        }
        return (react_2.default.createElement(eui_1.EuiInMemoryTable, { hasActions: true, columns: this.getTableColumns(role, availablePrivileges), search: {
                box: {
                    incremental: true,
                    placeholder: intl.formatMessage({
                        id: 'xpack.security.management.editRoles.privilegeSpaceTable.filterPlaceholder',
                        defaultMessage: 'Filter',
                    }),
                },
                onChange: (search) => {
                    this.setState({
                        searchTerm: search.queryText.toLowerCase(),
                    });
                },
            }, items: visibleTableItems }));
    }
}
exports.PrivilegeSpaceTable = react_1.injectI18n(PrivilegeSpaceTableUI);
