"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
// @ts-ignore
const content_disposition_1 = tslib_1.__importDefault(require("content-disposition"));
// @ts-ignore
const once_per_server_1 = require("../../lib/once_per_server");
const DEFAULT_TITLE = 'report';
const getTitle = (exportType, title) => `${title || DEFAULT_TITLE}.${exportType.jobContentExtension}`;
function getDocumentPayloadFn(server) {
    const exportTypesRegistry = server.plugins.reporting.exportTypesRegistry;
    function encodeContent(content, exportType) {
        switch (exportType.jobContentEncoding) {
            case 'base64':
                return Buffer.from(content, 'base64');
            default:
                return content;
        }
    }
    function getCompleted(output, jobType, title) {
        const exportType = exportTypesRegistry.get((item) => item.jobType === jobType);
        const filename = getTitle(exportType, title);
        return {
            statusCode: 200,
            content: encodeContent(output.content, exportType),
            contentType: output.content_type,
            headers: {
                'Content-Disposition': content_disposition_1.default(filename, { type: 'inline' }),
            },
        };
    }
    function getFailure(output) {
        return {
            statusCode: 500,
            content: {
                message: 'Reporting generation failed',
                reason: output.content,
            },
            contentType: 'application/json',
        };
    }
    function getIncomplete(status) {
        return {
            statusCode: 503,
            content: status,
            contentType: 'application/json',
            headers: {
                'retry-after': 30,
            },
        };
    }
    return function getDocumentPayload(doc) {
        const { status, output, jobtype: jobType, payload: { title } = { title: '' } } = doc._source;
        if (status === 'completed') {
            return getCompleted(output, jobType, title);
        }
        if (status === 'failed') {
            return getFailure(output);
        }
        // send a 503 indicating that the report isn't completed yet
        return getIncomplete(status);
    };
}
exports.getDocumentPayloadFactory = once_per_server_1.oncePerServer(getDocumentPayloadFn);
