"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const eui_1 = require("@elastic/eui");
const react_1 = require("@kbn/i18n/react");
const react_2 = tslib_1.__importStar(require("react"));
const notify_1 = require("ui/notify");
const url_1 = tslib_1.__importDefault(require("url"));
const reporting_client_1 = require("../lib/reporting_client");
class ReportingPanelContentUi extends react_2.Component {
    constructor(props) {
        super(props);
        this.renderGenerateReportButton = (isDisabled) => {
            return (react_2.default.createElement(eui_1.EuiButton, { className: "sharePanel__button", disabled: isDisabled, fill: true, onClick: this.createReportingJob, "data-test-subj": "generateReportButton", size: "s" },
                react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.reporting.panelContent.generateButtonLabel", defaultMessage: "Generate {reportingType}", values: { reportingType: this.prettyPrintReportingType() } })));
        };
        this.prettyPrintReportingType = () => {
            switch (this.props.reportType) {
                case 'printablePdf':
                    return 'PDF';
                case 'csv':
                    return 'CSV';
                case 'png':
                    return 'PNG';
                default:
                    return this.props.reportType;
            }
        };
        this.markAsStale = () => {
            if (!this.mounted) {
                return;
            }
            this.setState({ isStale: true });
        };
        this.isNotSaved = () => {
            return this.props.objectId === undefined || this.props.objectId === '';
        };
        this.setAbsoluteReportGenerationUrl = () => {
            if (!this.mounted) {
                return;
            }
            const absoluteUrl = ReportingPanelContentUi.getAbsoluteReportGenerationUrl(this.props);
            this.setState({ absoluteUrl });
        };
        this.createReportingJob = () => {
            const { intl } = this.props;
            return reporting_client_1.reportingClient
                .createReportingJob(this.props.reportType, this.props.getJobParams())
                .then(() => {
                notify_1.toastNotifications.addSuccess({
                    title: intl.formatMessage({
                        id: 'xpack.reporting.panelContent.successfullyQueuedReportNotificationTitle',
                        defaultMessage: 'Queued report for {objectType}',
                    }, { objectType: this.props.objectType }),
                    text: (react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.reporting.panelContent.successfullyQueuedReportNotificationDescription", defaultMessage: "Track its progress in Management" })),
                    'data-test-subj': 'queueReportSuccess',
                });
                this.props.onClose();
            })
                .catch((kfetchError) => {
                if (kfetchError.message === 'not exportable') {
                    return notify_1.toastNotifications.addWarning({
                        title: intl.formatMessage({
                            id: 'xpack.reporting.panelContent.whatCanBeExportedWarningTitle',
                            defaultMessage: 'Only saved {objectType} can be exported',
                        }, { objectType: this.props.objectType }),
                        text: (react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.reporting.panelContent.whatCanBeExportedWarningDescription", defaultMessage: "Please save your work first" })),
                    });
                }
                const defaultMessage = kfetchError.res.status === 403 ? (react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.reporting.panelContent.noPermissionToGenerateReportDescription", defaultMessage: "You don't have permission to generate this report." })) : (react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.reporting.panelContent.notification.cantReachServerDescription", defaultMessage: "Can't reach the server. Please try again." }));
                notify_1.toastNotifications.addDanger({
                    title: intl.formatMessage({
                        id: 'xpack.reporting.panelContent.notification.reportingErrorTitle',
                        defaultMessage: 'Reporting error',
                    }),
                    text: kfetchError.message || defaultMessage,
                    'data-test-subj': 'queueReportError',
                });
            });
        };
        this.state = {
            isStale: false,
            absoluteUrl: '',
            layoutId: '',
        };
    }
    static getDerivedStateFromProps(nextProps, prevState) {
        if (nextProps.layoutId !== prevState.layoutId) {
            return {
                ...prevState,
                absoluteUrl: ReportingPanelContentUi.getAbsoluteReportGenerationUrl(nextProps),
            };
        }
        return prevState;
    }
    componentWillUnmount() {
        window.removeEventListener('hashchange', this.markAsStale);
        window.removeEventListener('resize', this.setAbsoluteReportGenerationUrl);
        this.mounted = false;
    }
    componentDidMount() {
        this.mounted = true;
        window.addEventListener('hashchange', this.markAsStale, false);
        window.addEventListener('resize', this.setAbsoluteReportGenerationUrl);
    }
    render() {
        if (this.isNotSaved() || this.props.isDirty || this.state.isStale) {
            return (react_2.default.createElement(eui_1.EuiForm, { className: "sharePanelContent", "data-test-subj": "shareReportingForm" },
                react_2.default.createElement(eui_1.EuiFormRow, { helpText: react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.reporting.panelContent.saveWorkDescription", defaultMessage: "Please save your work before generating a report." }) }, this.renderGenerateReportButton(true))));
        }
        const reportMsg = (react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.reporting.panelContent.generationTimeDescription", defaultMessage: "{reportingType}s can take a minute or two to generate based upon the size of your {objectType}.", description: "Here 'reportingType' can be 'PDF' or 'CSV'", values: {
                reportingType: this.prettyPrintReportingType(),
                objectType: this.props.objectType,
            } }));
        return (react_2.default.createElement(eui_1.EuiForm, { className: "sharePanelContent", "data-test-subj": "shareReportingForm" },
            react_2.default.createElement(eui_1.EuiText, { size: "s" },
                react_2.default.createElement("p", null, reportMsg)),
            react_2.default.createElement(eui_1.EuiSpacer, { size: "s" }),
            this.props.options,
            this.renderGenerateReportButton(false),
            react_2.default.createElement(eui_1.EuiSpacer, { size: "s" }),
            react_2.default.createElement(eui_1.EuiText, { size: "s" },
                react_2.default.createElement("p", null,
                    react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.reporting.panelContent.howToCallGenerationDescription", defaultMessage: "Alternatively, copy this POST URL to call generation from outside Kibana or from Watcher." }))),
            react_2.default.createElement(eui_1.EuiSpacer, { size: "s" }),
            react_2.default.createElement(eui_1.EuiCopy, { textToCopy: this.state.absoluteUrl, anchorClassName: "sharePanel__copyAnchor" }, (copy) => (react_2.default.createElement(eui_1.EuiButton, { className: "sharePanel__button", onClick: copy, size: "s" },
                react_2.default.createElement(react_1.FormattedMessage, { id: "xpack.reporting.panelContent.copyUrlButtonLabel", defaultMessage: "Copy POST URL" }))))));
    }
}
ReportingPanelContentUi.getAbsoluteReportGenerationUrl = (props) => {
    const relativePath = reporting_client_1.reportingClient.getReportingJobPath(props.reportType, props.getJobParams());
    return url_1.default.resolve(window.location.href, relativePath);
};
exports.ReportingPanelContent = react_1.injectI18n(ReportingPanelContentUi);
