"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const lodash_1 = require("lodash");
const moment_1 = tslib_1.__importDefault(require("moment"));
const constants_1 = require("../constants");
const get_bucket_size_in_seconds_1 = require("./get_bucket_size_in_seconds");
// TODO: Break these function into seperate files and expand beyond just documnet count
// In the code below it looks like overkill to split these three functions out
// but in reality the create metrics functions will be different per node type.
const findLastFullBucket = (bucket, bucketSize, options) => {
    const { buckets } = bucket.timeseries;
    const to = moment_1.default.utc(options.timerange.to);
    return buckets.reduce((current, item) => {
        const itemKey = lodash_1.isNumber(item.key) ? item.key : parseInt(item.key, 10);
        const date = moment_1.default.utc(itemKey + bucketSize * 1000);
        if (!date.isAfter(to) && item.doc_count > 0) {
            return item;
        }
        return current;
    }, lodash_1.last(buckets));
};
const getMetricValueFromBucket = (type) => (bucket) => {
    const metric = bucket[type];
    return (metric && (metric.normalized_value || metric.value)) || 0;
};
function calculateMax(bucket, type) {
    const { buckets } = bucket.timeseries;
    return lodash_1.max(buckets.map(getMetricValueFromBucket(type))) || 0;
}
function calculateAvg(bucket, type) {
    const { buckets } = bucket.timeseries;
    return lodash_1.sum(buckets.map(getMetricValueFromBucket(type))) / buckets.length || 0;
}
function createNodeMetrics(options, node, bucket) {
    const { timerange, metric } = options;
    const bucketSize = get_bucket_size_in_seconds_1.getBucketSizeInSeconds(timerange.interval);
    const lastBucket = findLastFullBucket(bucket, bucketSize, options);
    if (!lastBucket) {
        throw new Error('Date histogram returned an empty set of buckets.');
    }
    return {
        name: metric.type,
        value: getMetricValueFromBucket(metric.type)(lastBucket),
        max: calculateMax(bucket, metric.type),
        avg: calculateAvg(bucket, metric.type),
    };
}
function createNodeItem(options, node, bucket) {
    const nodeDoc = lodash_1.get(node, ['nodeDetails', 'hits', 'hits', 0]);
    return {
        metric: createNodeMetrics(options, node, bucket),
        path: [{ value: node.key, label: lodash_1.get(nodeDoc, `_source.${constants_1.NAME_FIELDS[options.nodeType]}`) }],
    };
}
exports.createNodeItem = createNodeItem;
