"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const constate_1 = require("constate");
const react_1 = tslib_1.__importDefault(require("react"));
const react_apollo_1 = require("react-apollo");
const reselect_1 = require("reselect");
const memoize_1 = require("ui/utils/memoize");
const operation_status_1 = require("../../utils/operation_status");
const typed_constate_1 = require("../../utils/typed_constate");
const create_source_gql_query_1 = require("./create_source.gql_query");
const query_source_gql_query_1 = require("./query_source.gql_query");
const update_source_gql_query_1 = require("./update_source.gql_query");
const createContainerProps = memoize_1.memoizeLast((sourceId, apolloClient) => {
    const initialState = {
        operationStatusHistory: [],
        source: undefined,
    };
    const actions = typed_constate_1.inferActionMap()({
        ...operation_status_1.createStatusActions((updater) => (state) => ({
            ...state,
            operationStatusHistory: updater(state.operationStatusHistory),
        })),
    });
    const getDerivedIndexPattern = reselect_1.createSelector((state) => (state && state.source && state.source.status && state.source.status.indexFields) || [], (state) => (state &&
        state.source &&
        state.source.configuration &&
        state.source.configuration.logAlias) ||
        undefined, (state) => (state &&
        state.source &&
        state.source.configuration &&
        state.source.configuration.metricAlias) ||
        undefined, (indexFields, logAlias, metricAlias) => ({
        fields: indexFields,
        title: `${logAlias},${metricAlias}`,
    }));
    const selectors = typed_constate_1.inferSelectorMap()({
        ...operation_status_1.createStatusSelectors(({ operationStatusHistory }) => operationStatusHistory),
        getConfiguration: () => state => (state && state.source && state.source.configuration) || undefined,
        getSourceId: () => () => sourceId,
        getLogIndicesExist: () => state => (state && state.source && state.source.status && state.source.status.logIndicesExist) ||
            undefined,
        getMetricIndicesExist: () => state => (state && state.source && state.source.status && state.source.status.metricIndicesExist) ||
            undefined,
        getDerivedIndexPattern: () => getDerivedIndexPattern,
        getVersion: () => state => (state && state.source && state.source.version) || undefined,
        getExists: () => state => (state && state.source && !!state.source.version) || false,
    });
    const effects = typed_constate_1.inferEffectMap()({
        create: (sourceConfiguration) => ({ setState }) => {
            const variables = {
                sourceId,
                sourceConfiguration: {
                    name: sourceConfiguration.name,
                    description: sourceConfiguration.description,
                    metricAlias: sourceConfiguration.metricAlias,
                    logAlias: sourceConfiguration.logAlias,
                    fields: sourceConfiguration.fields
                        ? {
                            container: sourceConfiguration.fields.container,
                            host: sourceConfiguration.fields.host,
                            pod: sourceConfiguration.fields.pod,
                            tiebreaker: sourceConfiguration.fields.tiebreaker,
                            timestamp: sourceConfiguration.fields.timestamp,
                        }
                        : undefined,
                },
            };
            setState(actions.startOperation({ name: 'create', parameters: variables }));
            return apolloClient
                .mutate({
                mutation: create_source_gql_query_1.createSourceMutation,
                fetchPolicy: 'no-cache',
                variables,
            })
                .then(result => {
                setState(state => ({
                    ...actions.finishOperation({ name: 'create', parameters: variables })(state),
                    source: result.data ? result.data.createSource.source : state.source,
                }));
                return result;
            }, error => {
                setState(state => ({
                    ...actions.failOperation({ name: 'create', parameters: variables }, `${error}`)(state),
                }));
                throw error;
            });
        },
        load: () => ({ setState }) => {
            const variables = {
                sourceId,
            };
            setState(actions.startOperation({ name: 'load', parameters: variables }));
            return apolloClient
                .query({
                query: query_source_gql_query_1.sourceQuery,
                fetchPolicy: 'no-cache',
                variables,
            })
                .then(result => {
                setState(state => ({
                    ...actions.finishOperation({ name: 'load', parameters: variables })(state),
                    source: result.data.source,
                }));
                return result;
            }, error => {
                setState(state => ({
                    ...actions.failOperation({ name: 'load', parameters: variables }, `${error}`)(state),
                }));
                throw error;
            });
        },
        update: (changes) => ({ setState }) => {
            const variables = {
                sourceId,
                changes,
            };
            setState(actions.startOperation({ name: 'update', parameters: variables }));
            return apolloClient
                .mutate({
                mutation: update_source_gql_query_1.updateSourceMutation,
                fetchPolicy: 'no-cache',
                variables,
            })
                .then(result => {
                setState(state => ({
                    ...actions.finishOperation({ name: 'update', parameters: variables })(state),
                    source: result.data ? result.data.updateSource.source : state.source,
                }));
                return result;
            }, error => {
                setState(state => ({
                    ...actions.failOperation({ name: 'update', parameters: variables }, `${error}`)(state),
                }));
                throw error;
            });
        },
    });
    const onMount = props => {
        effects.load()(props);
    };
    return {
        actions,
        context: `source-${sourceId}`,
        effects,
        initialState,
        key: `source-${sourceId}`,
        onMount,
        selectors,
    };
});
exports.WithSource = ({ children }) => (react_1.default.createElement(react_apollo_1.ApolloConsumer, null, client => (react_1.default.createElement(constate_1.Container, Object.assign({}, createContainerProps('default', client)), ({ create, getConfiguration, getDerivedIndexPattern, getExists, getHasFailed, getIsInProgress, getLastFailureMessage, getLogIndicesExist, getMetricIndicesExist, getSourceId, getVersion, load, update, }) => children({
    create,
    configuration: getConfiguration(),
    derivedIndexPattern: getDerivedIndexPattern(),
    exists: getExists(),
    hasFailed: getHasFailed(),
    isLoading: getIsInProgress(),
    lastFailureMessage: getLastFailureMessage(),
    load,
    logIndicesExist: getLogIndicesExist(),
    metricIndicesExist: getMetricIndicesExist(),
    sourceId: getSourceId(),
    update,
    version: getVersion(),
})))));
