"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.shallowEqual = (a, b) => {
    if (a === b) {
        return true;
    }
    if (a.length !== b.length) {
        return false;
    }
    for (let i = 0; i < a.length; i++) {
        if (a[i] !== b[i]) {
            return false;
        }
    }
    return true;
};
const makeUid = () => 1e11 + Math.floor((1e12 - 1e11) * Math.random());
exports.selectReduce = (fun, previousValue) => (...inputs) => {
    // last-value memoizing version of this single line function:
    // (fun, previousValue) => (...inputs) => state => previousValue = fun(previousValue, ...inputs.map(input => input(state)))
    let argumentValues = [];
    let value = previousValue;
    let prevValue = previousValue;
    return (state) => {
        if (exports.shallowEqual(argumentValues, (argumentValues = inputs.map(input => input(state)))) &&
            value === prevValue) {
            return value;
        }
        prevValue = value;
        value = fun(prevValue, ...argumentValues);
        return value;
    };
};
exports.select = (fun) => (...inputs) => {
    // last-value memoizing version of this single line function:
    // fun => (...inputs) => state => fun(...inputs.map(input => input(state)))
    let argumentValues = [];
    let value;
    let actionId;
    return (state) => {
        const lastActionId = state.primaryUpdate.payload.uid;
        if (actionId === lastActionId ||
            exports.shallowEqual(argumentValues, (argumentValues = inputs.map(input => input(state))))) {
            return value;
        }
        value = fun(...argumentValues);
        actionId = lastActionId;
        return value;
    };
};
exports.createStore = (initialState, onChangeCallback) => {
    let currentState = initialState;
    let updater = (state) => state; // default: no side effect
    const getCurrentState = () => currentState;
    // const setCurrentState = newState => (currentState = newState);
    const setUpdater = (updaterFunction) => {
        updater = updaterFunction;
    };
    const commit = (type, payload, meta = { silent: false }) => {
        currentState = updater({
            ...currentState,
            primaryUpdate: {
                type,
                payload: { ...payload, uid: makeUid() },
            },
        });
        if (!meta.silent) {
            onChangeCallback({ type, state: currentState }, meta);
        }
    };
    const dispatch = (type, payload) => commit(type, payload);
    return { getCurrentState, setUpdater, commit, dispatch };
};
