/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */

import React from 'react';
import PropTypes from 'prop-types';
import { compose, withProps, withPropsOnChange } from 'recompose';
import { EuiForm, EuiTextArea, EuiButton, EuiButtonEmpty, EuiFormRow } from '@elastic/eui';
import { fromExpression, toExpression } from '@kbn/interpreter/common';
import { createStatefulPropHoc } from '../../components/enhance/stateful_prop';

export const AdvancedFailureComponent = props => {
  const {
    onValueChange,
    defaultValue,
    argExpression,
    updateArgExpression,
    resetErrorState,
    valid,
    argId,
  } = props;

  const valueChange = ev => {
    ev.preventDefault();

    resetErrorState(); // when setting a new value, attempt to reset the error state

    if (valid) {
      return onValueChange(fromExpression(argExpression.trim(), 'argument'));
    }
  };

  const confirmReset = ev => {
    ev.preventDefault();
    resetErrorState(); // when setting a new value, attempt to reset the error state
    onValueChange(fromExpression(defaultValue, 'argument'));
  };

  return (
    <EuiForm>
      <EuiFormRow id={argId} isInvalid={!valid} error="Invalid Expression">
        <EuiTextArea
          id={argId}
          isInvalid={!valid}
          value={argExpression}
          onChange={updateArgExpression}
          rows={3}
        />
      </EuiFormRow>
      <div>
        <EuiButton disabled={!valid} onClick={e => valueChange(e)} size="s" type="submit">
          Apply
        </EuiButton>
        {defaultValue && defaultValue.length && (
          <EuiButtonEmpty size="s" color="danger" onClick={confirmReset}>
            Reset
          </EuiButtonEmpty>
        )}
      </div>
    </EuiForm>
  );
};

AdvancedFailureComponent.propTypes = {
  defaultValue: PropTypes.string,
  onValueChange: PropTypes.func.isRequired,
  argExpression: PropTypes.string.isRequired,
  updateArgExpression: PropTypes.func.isRequired,
  resetErrorState: PropTypes.func.isRequired,
  valid: PropTypes.bool.isRequired,
  argId: PropTypes.string.isRequired,
};

export const AdvancedFailure = compose(
  withProps(({ argValue }) => ({
    argExpression: toExpression(argValue, 'argument'),
  })),
  createStatefulPropHoc('argExpression', 'updateArgExpression'),
  withPropsOnChange(['argExpression'], ({ argExpression }) => ({
    valid: (function() {
      try {
        fromExpression(argExpression, 'argument');
        return true;
      } catch (e) {
        return false;
      }
    })(),
  }))
)(AdvancedFailureComponent);

AdvancedFailure.propTypes = {
  argValue: PropTypes.any.isRequired,
};
