"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const boom_1 = tslib_1.__importDefault(require("boom"));
const lodash_1 = require("lodash");
class BackendFrameworkLib {
    constructor(adapter) {
        this.adapter = adapter;
        this.log = this.adapter.log;
        this.on = this.adapter.on.bind(this.adapter);
        this.exposeStaticDir = this.adapter.exposeStaticDir;
        this.internalUser = this.adapter.internalUser;
        this.validateConfig();
    }
    registerRoute(route) {
        this.adapter.registerRoute({
            ...route,
            handler: this.wrapRouteWithSecurity(route.handler, route.licenseRequired || [], route.requiredRoles),
        });
    }
    getSetting(setting) {
        return this.adapter.getSetting(`xpack.beats.${setting}`);
    }
    /**
     * Expired `null` happens when we have no xpack info
     */
    get license() {
        return {
            type: this.adapter.info ? this.adapter.info.license.type : 'unknown',
            expired: this.adapter.info ? this.adapter.info.license.expired : null,
        };
    }
    get securityIsEnabled() {
        return this.adapter.info ? this.adapter.info.security.enabled : false;
    }
    validateConfig() {
        const encryptionKey = this.adapter.getSetting('xpack.beats.encryptionKey');
        if (!encryptionKey) {
            this.adapter.log('Using a default encryption key for xpack.beats.encryptionKey. It is recommended that you set xpack.beats.encryptionKey in kibana.yml with a unique token');
        }
    }
    wrapRouteWithSecurity(handler, requiredLicense, requiredRoles) {
        return async (request, h) => {
            if (requiredLicense.length > 0 &&
                (this.license.expired || !requiredLicense.includes(this.license.type))) {
                return boom_1.default.forbidden(`Your ${this.license.type} license does not support this API or is expired. Please upgrade your license.`);
            }
            if (requiredRoles) {
                if (request.user.kind !== 'authenticated') {
                    return h.response().code(403);
                }
                if (request.user.kind === 'authenticated' &&
                    !request.user.roles.includes('superuser') &&
                    lodash_1.difference(requiredRoles, request.user.roles).length !== 0) {
                    return h.response().code(403);
                }
            }
            return await handler(request, h);
        };
    }
}
exports.BackendFrameworkLib = BackendFrameworkLib;
