"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lodash_1 = require("lodash");
const constants_1 = require("../../../../common/constants");
class ElasticsearchTagsAdapter {
    constructor(database) {
        this.database = database;
    }
    async getAll(user, ESQuery) {
        const params = {
            ignore: [404],
            _source: true,
            size: 10000,
            index: constants_1.INDEX_NAMES.BEATS,
            type: '_doc',
            body: {
                query: {
                    bool: {
                        must: {
                            term: {
                                type: 'tag',
                            },
                        },
                    },
                },
            },
        };
        if (ESQuery) {
            params.body.query = {
                ...params.body.query,
                ...ESQuery,
            };
        }
        const response = await this.database.search(user, params);
        const tags = lodash_1.get(response, 'hits.hits', []);
        return tags.map((tag) => ({ hasConfigurationBlocksTypes: [], ...tag._source.tag }));
    }
    async delete(user, tagIds) {
        const ids = tagIds.map(tag => tag);
        const params = {
            ignore: [404],
            index: constants_1.INDEX_NAMES.BEATS,
            type: '_doc',
            body: {
                query: {
                    terms: { 'beat.tags': tagIds },
                },
            },
        };
        const beatsResponse = await this.database.search(user, params);
        const beats = lodash_1.get(beatsResponse, 'hits.hits', []).map((beat) => beat._source.beat);
        const inactiveBeats = beats.filter(beat => beat.active === false);
        const activeBeats = beats.filter(beat => beat.active === true);
        if (activeBeats.length !== 0) {
            return false;
        }
        const beatIds = inactiveBeats.map((beat) => beat.id);
        // While we block tag deletion when on an active beat, we should remove from inactive
        const bulkInactiveBeatsUpdates = lodash_1.flatten(beatIds.map(beatId => {
            const script = `
        def beat = ctx._source.beat;
        if (beat.tags != null) {
          beat.tags.removeAll([params.tag]);
        }`;
            return lodash_1.flatten(ids.map(tagId => [
                { update: { _id: `beat:${beatId}` } },
                { script: { source: script.replace('          ', ''), params: { tagId } } },
            ]));
        }));
        const bulkTagsDelete = ids.map(tagId => ({ delete: { _id: `tag:${tagId}` } }));
        await this.database.bulk(user, {
            body: lodash_1.flatten([...bulkInactiveBeatsUpdates, ...bulkTagsDelete]),
            index: constants_1.INDEX_NAMES.BEATS,
            refresh: 'wait_for',
            type: '_doc',
        });
        return true;
    }
    async getTagsWithIds(user, tagIds) {
        if (tagIds.length === 0) {
            return [];
        }
        const ids = tagIds.map(tag => `tag:${tag}`);
        const params = {
            ignore: [404],
            _source: true,
            body: {
                ids,
            },
            index: constants_1.INDEX_NAMES.BEATS,
            type: '_doc',
        };
        const response = await this.database.mget(user, params);
        return lodash_1.get(response, 'docs', [])
            .filter((b) => b.found)
            .map((b) => ({
            hasConfigurationBlocksTypes: [],
            ...b._source.tag,
            id: b._id.replace('tag:', ''),
        }));
    }
    async upsertTag(user, tag) {
        const body = {
            tag,
            type: 'tag',
        };
        const params = {
            body,
            id: `tag:${tag.id}`,
            index: constants_1.INDEX_NAMES.BEATS,
            refresh: 'wait_for',
            type: '_doc',
        };
        const response = await this.database.index(user, params);
        return lodash_1.get(response, 'result');
    }
    async getWithoutConfigTypes(user, blockTypes) {
        const body = {
            query: {
                bool: {
                    filter: {
                        match: {
                            type: 'tag',
                        },
                    },
                    must_not: {
                        terms: { 'tag.hasConfigurationBlocksTypes': blockTypes },
                    },
                },
            },
        };
        const params = {
            body,
            index: constants_1.INDEX_NAMES.BEATS,
            ignore: [404],
            _source: true,
            size: 10000,
            type: '_doc',
        };
        const response = await this.database.search(user, params);
        const tags = lodash_1.get(response, 'hits.hits', []);
        return tags.map((tag) => ({ hasConfigurationBlocksTypes: [], ...tag._source.tag }));
    }
}
exports.ElasticsearchTagsAdapter = ElasticsearchTagsAdapter;
