"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const lodash_1 = require("lodash");
const v4_1 = tslib_1.__importDefault(require("uuid/v4"));
const constants_1 = require("../../../../common/constants");
class ElasticsearchConfigurationBlockAdapter {
    constructor(database) {
        this.database = database;
    }
    async getByIds(user, ids) {
        if (ids.length === 0) {
            return [];
        }
        const params = {
            ignore: [404],
            _source: true,
            size: 10000,
            index: constants_1.INDEX_NAMES.BEATS,
            type: '_doc',
            body: {
                ids: ids.map(id => `configuration_block:${id}`),
            },
        };
        const response = await this.database.search(user, params);
        const configs = lodash_1.get(response, 'hits.hits', []);
        return configs.map((tag) => ({ ...tag._source.tag, config: JSON.parse(tag._source.tag) }));
    }
    async getForTags(user, tagIds, page = 0, size = 100) {
        if (tagIds.length === 0) {
            return {
                page: 0,
                total: 0,
                blocks: [],
            };
        }
        const params = {
            ignore: [404],
            index: constants_1.INDEX_NAMES.BEATS,
            type: '_doc',
            body: {
                from: page === -1 ? undefined : page * size,
                size,
                query: {
                    terms: { 'configuration_block.tag': tagIds },
                },
            },
        };
        let response;
        if (page === -1) {
            response = await this.database.searchAll(user, params);
        }
        else {
            response = await this.database.search(user, params);
        }
        const configs = lodash_1.get(response, 'hits.hits', []);
        return {
            blocks: configs.map((block) => ({
                ...block._source.configuration_block,
                config: JSON.parse(block._source.configuration_block.config || '{}'),
            })),
            page,
            total: response.hits ? response.hits.total.value : 0,
        };
    }
    async delete(user, ids) {
        const result = await this.database.bulk(user, {
            body: ids.map(id => ({ delete: { _id: `configuration_block:${id}` } })),
            index: constants_1.INDEX_NAMES.BEATS,
            refresh: 'wait_for',
            type: '_doc',
        });
        if (result.errors) {
            if (result.items[0].result) {
                throw new Error(result.items[0].result);
            }
            throw new Error(result.items[0].index.error.reason);
        }
        return result.items.map((item) => {
            return {
                id: item.delete._id,
                success: item.delete.result === 'deleted',
                reason: item.delete.result !== 'deleted' ? item.delete.result : undefined,
            };
        });
    }
    async deleteForTags(user, tagIds) {
        const result = await this.database.deleteByQuery(user, {
            body: {
                query: {
                    terms: { 'configuration_block.tag': tagIds },
                },
            },
            index: constants_1.INDEX_NAMES.BEATS,
            type: '_doc',
        });
        if (result.failures.length > 0) {
            return {
                success: false,
                reason: result.failures[0],
            };
        }
        return {
            success: true,
        };
    }
    async create(user, configs) {
        const body = lodash_1.flatten(configs.map(config => {
            const id = config.id || v4_1.default();
            return [
                { index: { _id: `configuration_block:${id}` } },
                {
                    type: 'configuration_block',
                    configuration_block: { id, ...config, config: JSON.stringify(config.config) },
                },
            ];
        }));
        const result = await this.database.bulk(user, {
            body,
            index: constants_1.INDEX_NAMES.BEATS,
            refresh: 'wait_for',
            type: '_doc',
        });
        if (result.errors) {
            if (result.items[0].result) {
                throw new Error(result.items[0].result);
            }
            throw new Error(result.items[0].index.error.reason);
        }
        return result.items.map((item) => item.index._id);
    }
}
exports.ElasticsearchConfigurationBlockAdapter = ElasticsearchConfigurationBlockAdapter;
