"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lodash_1 = require("lodash");
const constants_1 = require("../../../../common/constants");
class ElasticsearchBeatsAdapter {
    constructor(database) {
        this.database = database;
    }
    async get(user, id) {
        const params = {
            id: `beat:${id}`,
            ignore: [404],
            index: constants_1.INDEX_NAMES.BEATS,
            type: '_doc',
        };
        const response = await this.database.get(user, params);
        if (!response.found) {
            return null;
        }
        const beat = lodash_1.get(response, '_source.beat');
        beat.tags = beat.tags || [];
        return beat;
    }
    async insert(user, beat) {
        const body = {
            beat,
            type: 'beat',
        };
        await this.database.index(user, {
            body,
            id: `beat:${beat.id}`,
            index: constants_1.INDEX_NAMES.BEATS,
            refresh: 'wait_for',
            type: '_doc',
        });
    }
    async update(user, beat) {
        const body = {
            beat,
            type: 'beat',
        };
        const params = {
            body,
            id: `beat:${beat.id}`,
            index: constants_1.INDEX_NAMES.BEATS,
            refresh: 'wait_for',
            type: '_doc',
        };
        await this.database.index(user, params);
    }
    async getWithIds(user, beatIds) {
        const ids = beatIds.map(beatId => `beat:${beatId}`);
        const params = {
            body: {
                ids,
            },
            index: constants_1.INDEX_NAMES.BEATS,
            type: '_doc',
        };
        const response = await this.database.mget(user, params);
        return lodash_1.get(response, 'docs', [])
            .filter((b) => b.found)
            .map((b) => ({ tags: [], ...b._source.beat }));
    }
    async getAllWithTags(user, tagIds) {
        const params = {
            ignore: [404],
            index: constants_1.INDEX_NAMES.BEATS,
            type: '_doc',
            body: {
                query: {
                    terms: { 'beat.tags': tagIds },
                },
            },
        };
        const response = await this.database.search(user, params);
        const beats = lodash_1.get(response, 'hits.hits', []);
        if (beats.length === 0) {
            return [];
        }
        return beats.map((beat) => ({
            tags: [],
            ...lodash_1.omit(beat._source.beat, ['access_token']),
        }));
    }
    async getBeatWithToken(user, enrollmentToken) {
        const params = {
            ignore: [404],
            index: constants_1.INDEX_NAMES.BEATS,
            type: '_doc',
            body: {
                query: {
                    match: { 'beat.enrollment_token': enrollmentToken },
                },
            },
        };
        const response = await this.database.search(user, params);
        const beats = lodash_1.get(response, 'hits.hits', []);
        if (beats.length === 0) {
            return null;
        }
        return lodash_1.omit(lodash_1.get({ tags: [], ...beats[0] }, '_source.beat'), [
            'access_token',
        ]);
    }
    async getAll(user, ESQuery) {
        const params = {
            index: constants_1.INDEX_NAMES.BEATS,
            size: 10000,
            ignore: [404],
            type: '_doc',
            body: {
                query: {
                    bool: {
                        must: {
                            term: {
                                type: 'beat',
                            },
                        },
                    },
                },
            },
        };
        if (ESQuery) {
            params.body.query = {
                ...params.body.query,
                ...ESQuery,
            };
        }
        let response;
        try {
            response = await this.database.search(user, params);
        }
        catch (e) {
            // TODO something
        }
        if (!response) {
            return [];
        }
        const beats = lodash_1.get(response, 'hits.hits', []);
        return beats.map((beat) => ({
            tags: [],
            ...lodash_1.omit(beat._source.beat, ['access_token']),
        }));
    }
    async removeTagsFromBeats(user, removals) {
        const body = lodash_1.flatten(removals.map(({ beatId, tag }) => {
            const script = `
          def beat = ctx._source.beat;
          if (beat.tags != null) {
            beat.tags.removeAll([params.tag]);
          }`;
            return [
                { update: { _id: `beat:${beatId}` } },
                { script: { source: script.replace('          ', ''), params: { tag } } },
            ];
        }));
        const response = await this.database.bulk(user, {
            body,
            index: constants_1.INDEX_NAMES.BEATS,
            refresh: 'wait_for',
            type: '_doc',
        });
        return lodash_1.get(response, 'items', []).map((item, resultIdx) => ({
            idxInRequest: removals[resultIdx].idxInRequest,
            result: item.update.result,
            status: item.update.status,
        }));
    }
    async assignTagsToBeats(user, assignments) {
        const body = lodash_1.flatten(assignments.map(({ beatId, tag }) => {
            const script = `
          def beat = ctx._source.beat;
          if (beat.tags == null) {
            beat.tags = [];
          }
          if (!beat.tags.contains(params.tag)) {
            beat.tags.add(params.tag);
          }`;
            return [
                { update: { _id: `beat:${beatId}` } },
                { script: { source: script.replace('          ', ''), params: { tag } } },
            ];
        }));
        const response = await this.database.bulk(user, {
            body,
            index: constants_1.INDEX_NAMES.BEATS,
            refresh: 'wait_for',
            type: '_doc',
        });
        // console.log(response.items[0].update.error);
        return lodash_1.get(response, 'items', []).map((item, resultIdx) => ({
            idxInRequest: assignments[resultIdx].idxInRequest,
            result: item.update.result,
            status: item.update.status,
        }));
    }
}
exports.ElasticsearchBeatsAdapter = ElasticsearchBeatsAdapter;
