"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const constants_1 = require("../../../../../common/constants");
const get_bucket_size_1 = require("../../../helpers/get_bucket_size");
function timeseriesFetcher({ serviceName, transactionType, transactionName, setup }) {
    const { start, end, esFilterQuery, client, config } = setup;
    const { intervalString } = get_bucket_size_1.getBucketSize(start, end, 'auto');
    const filter = [
        { term: { [constants_1.PROCESSOR_EVENT]: 'transaction' } },
        { term: { [constants_1.SERVICE_NAME]: serviceName } },
        {
            range: {
                '@timestamp': {
                    gte: start,
                    lte: end,
                    format: 'epoch_millis'
                }
            }
        }
    ];
    if (transactionType) {
        filter.push({ term: { [constants_1.TRANSACTION_TYPE]: transactionType } });
    }
    if (esFilterQuery) {
        filter.push(esFilterQuery);
    }
    const params = {
        index: config.get('apm_oss.transactionIndices'),
        body: {
            size: 0,
            query: { bool: { filter } },
            aggs: {
                response_times: {
                    date_histogram: {
                        field: '@timestamp',
                        interval: intervalString,
                        min_doc_count: 0,
                        extended_bounds: { min: start, max: end }
                    },
                    aggs: {
                        avg: { avg: { field: constants_1.TRANSACTION_DURATION } },
                        pct: {
                            percentiles: { field: constants_1.TRANSACTION_DURATION, percents: [95, 99] }
                        }
                    }
                },
                overall_avg_duration: { avg: { field: constants_1.TRANSACTION_DURATION } },
                transaction_results: {
                    terms: { field: constants_1.TRANSACTION_RESULT, missing: '' },
                    aggs: {
                        timeseries: {
                            date_histogram: {
                                field: '@timestamp',
                                interval: intervalString,
                                min_doc_count: 0,
                                extended_bounds: { min: start, max: end }
                            }
                        }
                    }
                }
            }
        }
    };
    if (transactionName) {
        params.body.query.bool.must = [
            { term: { [`${constants_1.TRANSACTION_NAME}.keyword`]: transactionName } }
        ];
    }
    return client('search', params);
}
exports.timeseriesFetcher = timeseriesFetcher;
