"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const ts_optchain_1 = require("ts-optchain");
const constants_1 = require("../../../common/constants");
async function getErrorGroups({ serviceName, sortField, sortDirection = 'desc', setup }) {
    const { start, end, esFilterQuery, client, config } = setup;
    const params = {
        index: config.get('apm_oss.errorIndices'),
        body: {
            size: 0,
            query: {
                bool: {
                    filter: [
                        { term: { [constants_1.SERVICE_NAME]: serviceName } },
                        { term: { [constants_1.PROCESSOR_EVENT]: 'error' } },
                        {
                            range: {
                                '@timestamp': {
                                    gte: start,
                                    lte: end,
                                    format: 'epoch_millis'
                                }
                            }
                        }
                    ]
                }
            },
            aggs: {
                error_groups: {
                    terms: {
                        field: constants_1.ERROR_GROUP_ID,
                        size: 500,
                        order: { _count: sortDirection }
                    },
                    aggs: {
                        sample: {
                            top_hits: {
                                _source: [
                                    constants_1.ERROR_LOG_MESSAGE,
                                    constants_1.ERROR_EXC_MESSAGE,
                                    constants_1.ERROR_EXC_HANDLED,
                                    constants_1.ERROR_CULPRIT,
                                    constants_1.ERROR_GROUP_ID,
                                    '@timestamp'
                                ],
                                sort: [{ '@timestamp': 'desc' }],
                                size: 1
                            }
                        }
                    }
                }
            }
        }
    };
    if (esFilterQuery) {
        params.body.query.bool.filter.push(esFilterQuery);
    }
    // sort buckets by last occurrence of error
    if (sortField === 'latestOccurrenceAt') {
        params.body.aggs.error_groups.terms.order = {
            max_timestamp: sortDirection
        };
        params.body.aggs.error_groups.aggs.max_timestamp = {
            max: { field: '@timestamp' }
        };
    }
    const resp = await client('search', params);
    const hits = ts_optchain_1.oc(resp)
        .aggregations.error_groups.buckets([])
        .map(bucket => {
        const source = bucket.sample.hits.hits[0]._source;
        const message = ts_optchain_1.oc(source).error.log.message() || ts_optchain_1.oc(source).error.exception.message();
        return {
            message,
            occurrenceCount: bucket.doc_count,
            culprit: ts_optchain_1.oc(source).error.culprit(),
            groupId: ts_optchain_1.oc(source).error.grouping_key(),
            latestOccurrenceAt: source['@timestamp'],
            handled: ts_optchain_1.oc(source).error.exception.handled()
        };
    });
    return hits;
}
exports.getErrorGroups = getErrorGroups;
