"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lodash_1 = require("lodash");
const ts_optchain_1 = require("ts-optchain");
const constants_1 = require("../../../common/constants");
const get_transaction_1 = require("../transactions/get_transaction");
// TODO: rename from "getErrorGroup"  to "getErrorGroupSample" (since a single error is returned, not an errorGroup)
async function getErrorGroup({ serviceName, groupId, setup }) {
    const { start, end, esFilterQuery, client, config } = setup;
    const filter = [
        { term: { [constants_1.SERVICE_NAME]: serviceName } },
        { term: { [constants_1.ERROR_GROUP_ID]: groupId } },
        {
            range: {
                '@timestamp': {
                    gte: start,
                    lte: end,
                    format: 'epoch_millis'
                }
            }
        }
    ];
    if (esFilterQuery) {
        filter.push(esFilterQuery);
    }
    const params = {
        index: config.get('apm_oss.errorIndices'),
        body: {
            size: 1,
            query: {
                bool: {
                    filter,
                    should: [{ term: { [constants_1.TRANSACTION_SAMPLED]: true } }]
                }
            },
            sort: [
                { _score: 'desc' },
                { '@timestamp': { order: 'desc' } } // sort by timestamp to get the most recent error
            ]
        }
    };
    const resp = await client('search', params);
    const error = ts_optchain_1.oc(resp).hits.hits[0]._source();
    const transactionId = ts_optchain_1.oc(error).transaction.id();
    const traceId = lodash_1.get(error, 'trace.id'); // cannot use oc because 'trace' doesn't exist on v1 errors
    let transaction;
    if (transactionId) {
        transaction = await get_transaction_1.getTransaction(transactionId, traceId, setup);
    }
    return {
        transaction,
        error,
        occurrencesCount: resp.hits.total
    };
}
exports.getErrorGroup = getErrorGroup;
