"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const constants_1 = require("../../../../common/constants");
async function getBuckets({ serviceName, groupId, bucketSize, setup }) {
    const { start, end, esFilterQuery, client, config } = setup;
    const filter = [
        { term: { [constants_1.PROCESSOR_EVENT]: 'error' } },
        { term: { [constants_1.SERVICE_NAME]: serviceName } },
        {
            range: {
                '@timestamp': {
                    gte: start,
                    lte: end,
                    format: 'epoch_millis'
                }
            }
        }
    ];
    if (groupId) {
        filter.push({ term: { [constants_1.ERROR_GROUP_ID]: groupId } });
    }
    if (esFilterQuery) {
        filter.push(esFilterQuery);
    }
    const params = {
        index: config.get('apm_oss.errorIndices'),
        body: {
            size: 0,
            query: {
                bool: {
                    filter
                }
            },
            aggs: {
                distribution: {
                    histogram: {
                        field: '@timestamp',
                        min_doc_count: 0,
                        interval: bucketSize,
                        extended_bounds: {
                            min: start,
                            max: end
                        }
                    }
                }
            }
        }
    };
    const resp = await client('search', params);
    const buckets = resp.aggregations.distribution.buckets.map(bucket => ({
        key: bucket.key,
        count: bucket.doc_count
    }));
    return {
        totalHits: resp.hits.total,
        buckets
    };
}
exports.getBuckets = getBuckets;
