"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License;
 * you may not use this file except in compliance with the Elastic License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const eui_1 = require("@elastic/eui");
const i18n_1 = require("@kbn/i18n");
const idx_1 = tslib_1.__importDefault(require("idx"));
const react_1 = tslib_1.__importDefault(require("react"));
const url_helpers_1 = require("x-pack/plugins/apm/public/components/shared/Links/url_helpers");
const DiscoverTransactionLink_1 = require("../Links/DiscoverLinks/DiscoverTransactionLink");
const QueryWithIndexPattern_1 = require("../Links/DiscoverLinks/QueryWithIndexPattern");
function getInfraMetricsQuery(transaction) {
    const plus5 = new Date(transaction['@timestamp']);
    const minus5 = new Date(transaction['@timestamp']);
    plus5.setMinutes(plus5.getMinutes() + 5);
    minus5.setMinutes(minus5.getMinutes() - 5);
    return {
        from: minus5.getTime(),
        to: plus5.getTime()
    };
}
function ActionMenuButton({ onClick }) {
    return (react_1.default.createElement(eui_1.EuiButtonEmpty, { iconType: "arrowDown", iconSide: "right", onClick: onClick }, i18n_1.i18n.translate('xpack.apm.transactionActionMenu.actionsButtonLabel', {
        defaultMessage: 'Actions'
    })));
}
class TransactionActionMenu extends react_1.default.Component {
    constructor() {
        super(...arguments);
        this.state = {
            isOpen: false
        };
        this.toggle = () => {
            this.setState(state => ({ isOpen: !state.isOpen }));
        };
        this.close = () => {
            this.setState({ isOpen: false });
        };
    }
    getInfraActions() {
        const { transaction, location } = this.props;
        const hostName = idx_1.default(transaction, _ => _.context.system.hostname);
        const podId = idx_1.default(transaction, _ => _.kubernetes.pod.uid);
        const containerId = idx_1.default(transaction, _ => _.docker.container.id);
        const pathname = '/app/infra';
        const time = new Date(transaction['@timestamp']).getTime();
        const infraMetricsQuery = getInfraMetricsQuery(transaction);
        return [
            {
                icon: 'loggingApp',
                label: i18n_1.i18n.translate('xpack.apm.transactionActionMenu.showPodLogsLinkLabel', { defaultMessage: 'Show pod logs' }),
                target: podId,
                hash: `/link-to/pod-logs/${podId}`,
                query: { time }
            },
            {
                icon: 'loggingApp',
                label: i18n_1.i18n.translate('xpack.apm.transactionActionMenu.showContainerLogsLinkLabel', { defaultMessage: 'Show container logs' }),
                target: containerId,
                hash: `/link-to/container-logs/${containerId}`,
                query: { time }
            },
            {
                icon: 'loggingApp',
                label: i18n_1.i18n.translate('xpack.apm.transactionActionMenu.showHostLogsLinkLabel', { defaultMessage: 'Show host logs' }),
                target: hostName,
                hash: `/link-to/host-logs/${hostName}`,
                query: { time }
            },
            {
                icon: 'infraApp',
                label: i18n_1.i18n.translate('xpack.apm.transactionActionMenu.showPodMetricsLinkLabel', { defaultMessage: 'Show pod metrics' }),
                target: podId,
                hash: `/link-to/pod-detail/${podId}`,
                query: infraMetricsQuery
            },
            {
                icon: 'infraApp',
                label: i18n_1.i18n.translate('xpack.apm.transactionActionMenu.showContainerMetricsLinkLabel', { defaultMessage: 'Show container metrics' }),
                target: containerId,
                hash: `/link-to/container-detail/${containerId}`,
                query: infraMetricsQuery
            },
            {
                icon: 'infraApp',
                label: i18n_1.i18n.translate('xpack.apm.transactionActionMenu.showHostMetricsLinkLabel', { defaultMessage: 'Show host metrics' }),
                target: hostName,
                hash: `/link-to/host-detail/${hostName}`,
                query: infraMetricsQuery
            }
        ]
            .filter(({ target }) => Boolean(target))
            .map(({ icon, label, hash, query }, index) => {
            const href = url_helpers_1.getKibanaHref({
                location,
                pathname,
                hash,
                query: query // TODO: differentiate between APM ui query args, and external query args
            });
            return (react_1.default.createElement(eui_1.EuiContextMenuItem, { icon: icon, href: href, key: index },
                react_1.default.createElement(eui_1.EuiFlexGroup, { gutterSize: "s" },
                    react_1.default.createElement(eui_1.EuiFlexItem, null,
                        react_1.default.createElement(eui_1.EuiLink, null, label)),
                    react_1.default.createElement(eui_1.EuiFlexItem, { grow: false },
                        react_1.default.createElement(eui_1.EuiIcon, { type: "popout" })))));
        });
    }
    render() {
        const { transaction, location } = this.props;
        return (react_1.default.createElement(QueryWithIndexPattern_1.QueryWithIndexPattern, { query: DiscoverTransactionLink_1.getDiscoverQuery(transaction) }, query => {
            const discoverTransactionHref = url_helpers_1.getKibanaHref({
                location,
                pathname: '/app/kibana',
                hash: '/discover',
                query
            });
            const items = [
                ...this.getInfraActions(),
                react_1.default.createElement(eui_1.EuiContextMenuItem, { icon: "discoverApp", href: discoverTransactionHref, key: "discover-transaction" },
                    react_1.default.createElement(eui_1.EuiFlexGroup, { gutterSize: "s" },
                        react_1.default.createElement(eui_1.EuiFlexItem, null,
                            react_1.default.createElement(eui_1.EuiLink, null, i18n_1.i18n.translate('xpack.apm.transactionActionMenu.viewSampleDocumentLinkLabel', {
                                defaultMessage: 'View sample document'
                            }))),
                        react_1.default.createElement(eui_1.EuiFlexItem, { grow: false },
                            react_1.default.createElement(eui_1.EuiIcon, { type: "popout" }))))
            ];
            return (react_1.default.createElement(eui_1.EuiPopover, { id: "transactionActionMenu", button: react_1.default.createElement(ActionMenuButton, { onClick: this.toggle }), isOpen: this.state.isOpen, closePopover: this.close, anchorPosition: "downRight", panelPaddingSize: "none" },
                react_1.default.createElement(eui_1.EuiContextMenuPanel, { items: items, title: i18n_1.i18n.translate('xpack.apm.transactionActionMenu.actionsLabel', { defaultMessage: 'Actions' }) })));
        }));
    }
}
exports.TransactionActionMenu = TransactionActionMenu;
